/*
 * Copyright 2010-2020 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.fir.analysis.checkers.extra

import ksp.com.intellij.lang.LighterASTNode
import ksp.com.intellij.lang.PsiBuilder
import ksp.com.intellij.psi.PsiElement
import ksp.org.jetbrains.kotlin.KtLightSourceElement
import ksp.org.jetbrains.kotlin.KtNodeTypes
import ksp.org.jetbrains.kotlin.KtPsiSourceElement
import ksp.org.jetbrains.kotlin.diagnostics.DiagnosticReporter
import ksp.org.jetbrains.kotlin.diagnostics.reportOn
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.MppCheckerKind
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.context.CheckerContext
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.expression.FirStringConcatenationCallChecker
import ksp.org.jetbrains.kotlin.fir.analysis.diagnostics.FirErrors.REDUNDANT_SINGLE_EXPRESSION_STRING_TEMPLATE
import ksp.org.jetbrains.kotlin.fir.expressions.FirStatement
import ksp.org.jetbrains.kotlin.fir.expressions.FirStringConcatenationCall
import ksp.org.jetbrains.kotlin.fir.expressions.arguments
import ksp.org.jetbrains.kotlin.fir.analysis.checkers.fullyExpandedClassId
import ksp.org.jetbrains.kotlin.fir.types.resolvedType
import ksp.org.jetbrains.kotlin.name.StandardClassIds
import ksp.org.jetbrains.kotlin.psi.KtStringTemplateExpression
import ksp.org.jetbrains.kotlin.util.getChildren

object RedundantSingleExpressionStringTemplateChecker : FirStringConcatenationCallChecker(MppCheckerKind.Common) {
    context(context: CheckerContext, reporter: DiagnosticReporter)
    override fun check(expression: FirStringConcatenationCall) {
        for (argumentExpression in expression.arguments) {
            if (argumentExpression.resolvedType.fullyExpandedClassId(context.session) == StandardClassIds.String &&
                argumentExpression.stringParentChildrenCount() == 1 // there is no more children in original string template
            ) {
                reporter.reportOn(argumentExpression.source, REDUNDANT_SINGLE_EXPRESSION_STRING_TEMPLATE)
            }
        }
    }

    private fun FirStatement.stringParentChildrenCount(): Int? {
        return when (val source = source) {
            is KtPsiSourceElement -> source.psi.stringParentChildrenCount()
            is KtLightSourceElement -> source.lighterASTNode.stringParentChildrenCount(source)
            null -> null
        }
    }

    private fun PsiElement.stringParentChildrenCount(): Int? {
        return when (val parent = parent) {
            is KtStringTemplateExpression -> return parent.children.size
            null -> null
            else -> parent.stringParentChildrenCount()
        }
    }

    private fun LighterASTNode.stringParentChildrenCount(source: KtLightSourceElement): Int? {
        val parent = source.treeStructure.getParent(this)
        return if (parent != null && parent.tokenType == KtNodeTypes.STRING_TEMPLATE) {
            val childrenOfParent = parent.getChildren(source.treeStructure)
            childrenOfParent.filter { it is PsiBuilder.Marker }.size
        } else {
            parent?.stringParentChildrenCount(source)
        }
    }
}
