/*
 * Copyright 2010-2024 JetBrains s.r.o. and Kotlin Programming Language contributors.
 * Use of this source code is governed by the Apache 2.0 license that can be found in the license/LICENSE.txt file.
 */

package ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.projectStructure

import ksp.com.intellij.openapi.project.Project
import ksp.com.intellij.openapi.util.Disposer
import ksp.org.jetbrains.kotlin.analysis.api.platform.declarations.createAnnotationResolver
import ksp.org.jetbrains.kotlin.analysis.api.platform.projectStructure.KotlinCompilerPluginsProvider
import ksp.org.jetbrains.kotlin.analysis.api.platform.projectStructure.KaResolutionScopeProvider
import ksp.org.jetbrains.kotlin.analysis.api.projectStructure.KaSourceModule
import ksp.org.jetbrains.kotlin.analysis.api.resolve.extensions.KaResolveExtensionProvider
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.caches.FirThreadSafeCachesFactory
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.compile.CodeFragmentScopeProvider
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.diagnostics.LLCheckersFactory
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.providers.*
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.resolve.extensions.LLFirNonEmptyResolveExtensionTool
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.resolve.extensions.LLFirResolveExtensionTool
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.sessions.LLFirSession
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.util.FirElementFinder
import ksp.org.jetbrains.kotlin.analysis.low.level.api.fir.util.LLFirExceptionHandler
import ksp.org.jetbrains.kotlin.config.LanguageVersionSettings
import ksp.org.jetbrains.kotlin.fir.FirExceptionHandler
import ksp.org.jetbrains.kotlin.fir.FirPrivateVisibleFromDifferentModuleExtension
import ksp.org.jetbrains.kotlin.fir.FirSession
import ksp.org.jetbrains.kotlin.fir.SessionConfiguration
import ksp.org.jetbrains.kotlin.fir.caches.FirCachesFactory
import ksp.org.jetbrains.kotlin.fir.declarations.SealedClassInheritorsProvider
import ksp.org.jetbrains.kotlin.fir.extensions.FirExtensionRegistrar
import ksp.org.jetbrains.kotlin.fir.extensions.FirExtensionRegistrarAdapter
import ksp.org.jetbrains.kotlin.fir.extensions.FirPredicateBasedProvider
import ksp.org.jetbrains.kotlin.fir.extensions.FirRegisteredPluginAnnotations
import ksp.org.jetbrains.kotlin.fir.resolve.providers.FirSymbolProvider
import ksp.org.jetbrains.kotlin.fir.resolve.providers.impl.FirCompositeSymbolProvider
import ksp.org.jetbrains.kotlin.fir.scopes.FirLookupDefaultStarImportsInSourcesSettingHolder
import ksp.org.jetbrains.kotlin.fir.session.FirSessionConfigurator

@SessionConfiguration
internal fun LLFirSession.registerIdeComponents(project: Project, languageVersionSettings: LanguageVersionSettings) {
    register(FirCachesFactory::class, FirThreadSafeCachesFactory(project))
    register(SealedClassInheritorsProvider::class, LLSealedInheritorsProvider(project))
    register(FirExceptionHandler::class, LLFirExceptionHandler)
    register(CodeFragmentScopeProvider::class, CodeFragmentScopeProvider(this))
    register(FirElementFinder::class, FirElementFinder())
    register(FirPrivateVisibleFromDifferentModuleExtension::class, LLFirPrivateVisibleFromDifferentModuleExtension(this))
    register(
        FirLookupDefaultStarImportsInSourcesSettingHolder::class,
        createLookupDefaultStarImportsInSourcesSettingHolder(languageVersionSettings)
    )
    register(LLCheckersFactory::class, LLCheckersFactory(this))
    registerResolveExtensionTool()
}

@SessionConfiguration
private fun LLFirSession.registerResolveExtensionTool() {
    val resolveExtensionTool = createResolveExtensionTool() ?: return

    // `KaResolveExtension`s are disposables meant to be tied to the lifetime of the `LLFirSession`.
    resolveExtensionTool.extensions.forEach { Disposer.register(requestDisposable(), it) }

    register(LLFirResolveExtensionTool::class, resolveExtensionTool)
}

private fun LLFirSession.createResolveExtensionTool(): LLFirResolveExtensionTool? {
    val extensions = KaResolveExtensionProvider.provideExtensionsFor(ktModule)
    if (extensions.isEmpty()) return null
    return LLFirNonEmptyResolveExtensionTool(this, extensions)
}


internal inline fun createCompositeSymbolProvider(
    session: FirSession,
    createSubProviders: MutableList<FirSymbolProvider>.() -> Unit
): FirCompositeSymbolProvider =
    FirCompositeSymbolProvider(session, buildList(createSubProviders))

@SessionConfiguration
internal fun FirSession.registerCompilerPluginExtensions(project: Project, module: KaSourceModule) {
    FirSessionConfigurator(this).apply {
        FirExtensionRegistrarAdapter.getInstances(project).forEach(::applyExtensionRegistrar)

        KotlinCompilerPluginsProvider.getInstance(project)
            ?.getRegisteredExtensions(module, FirExtensionRegistrarAdapter)
            ?.forEach(::applyExtensionRegistrar)
    }.configure()
}

private fun FirSessionConfigurator.applyExtensionRegistrar(registrar: FirExtensionRegistrarAdapter) {
    registerExtensions((registrar as FirExtensionRegistrar).configure())
}

@SessionConfiguration
internal fun LLFirSession.registerCompilerPluginServices(
    project: Project,
    module: KaSourceModule
) {
    val projectWithDependenciesScope = KaResolutionScopeProvider.getInstance(project).getResolutionScope(module)
    val annotationsResolver = project.createAnnotationResolver(projectWithDependenciesScope)

    // We need FirRegisteredPluginAnnotations and FirPredicateBasedProvider during extensions' registration process
    register(FirRegisteredPluginAnnotations::class, LLFirIdeRegisteredPluginAnnotations(this, annotationsResolver))
    register(FirPredicateBasedProvider::class, LLFirIdePredicateBasedProvider(this, annotationsResolver))
}
