/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner;

import com.google.api.core.ApiFuture;
import com.google.api.core.ApiFutureCallback;
import com.google.api.core.ApiFutures;
import com.google.api.gax.core.CredentialsProvider;
import com.google.api.gax.core.FixedCredentialsProvider;
import com.google.api.gax.core.NoCredentialsProvider;
import com.google.api.gax.rpc.PermissionDeniedException;
import com.google.auth.Credentials;
import com.google.cloud.monitoring.v3.MetricServiceClient;
import com.google.cloud.monitoring.v3.MetricServiceSettings;
import com.google.cloud.spanner.SpannerCloudMonitoringExporterUtils;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.monitoring.v3.CreateTimeSeriesRequest;
import com.google.monitoring.v3.ProjectName;
import com.google.monitoring.v3.TimeSeries;
import com.google.protobuf.Empty;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.metrics.InstrumentType;
import io.opentelemetry.sdk.metrics.data.AggregationTemporality;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.export.MetricExporter;
import io.opentelemetry.sdk.resources.Resource;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;

class SpannerCloudMonitoringExporter
implements MetricExporter {
    private static final Logger logger = Logger.getLogger(SpannerCloudMonitoringExporter.class.getName());
    private static final int EXPORT_BATCH_SIZE_LIMIT = 200;
    private final AtomicBoolean spannerExportFailureLogged = new AtomicBoolean(false);
    private final AtomicBoolean lastExportSkippedData = new AtomicBoolean(false);
    private final MetricServiceClient client;
    private final String spannerProjectId;

    static SpannerCloudMonitoringExporter create(String projectId, @Nullable Credentials credentials, @Nullable String monitoringHost) throws IOException {
        MetricServiceSettings.Builder settingsBuilder = MetricServiceSettings.newBuilder();
        Object credentialsProvider = credentials == null ? NoCredentialsProvider.create() : FixedCredentialsProvider.create((Credentials)credentials);
        settingsBuilder.setCredentialsProvider((CredentialsProvider)credentialsProvider);
        if (monitoringHost != null) {
            settingsBuilder.setEndpoint(monitoringHost);
        }
        Duration timeout = Duration.ofMinutes(1L);
        settingsBuilder.createServiceTimeSeriesSettings().setSimpleTimeoutNoRetriesDuration(timeout);
        return new SpannerCloudMonitoringExporter(projectId, MetricServiceClient.create((MetricServiceSettings)settingsBuilder.build()));
    }

    @VisibleForTesting
    SpannerCloudMonitoringExporter(String projectId, MetricServiceClient client) {
        this.client = client;
        this.spannerProjectId = projectId;
    }

    public CompletableResultCode export(@Nonnull Collection<MetricData> collection) {
        if (this.client.isShutdown()) {
            logger.log(Level.WARNING, "Exporter is shut down");
            return CompletableResultCode.ofFailure();
        }
        return this.exportSpannerClientMetrics(collection);
    }

    private CompletableResultCode exportSpannerClientMetrics(Collection<MetricData> collection) {
        List<TimeSeries> spannerTimeSeries;
        List<MetricData> spannerMetricData = collection.stream().collect(Collectors.toList());
        boolean mustFilter = false;
        if (spannerMetricData.stream().map(metricData -> metricData.getResource()).anyMatch(this::shouldSkipPointDataDueToProjectId)) {
            logger.log(Level.WARNING, "Some metric data contain a different projectId. These will be skipped.");
            mustFilter = true;
        }
        if (mustFilter) {
            spannerMetricData = spannerMetricData.stream().filter(this::shouldSkipMetricData).collect(Collectors.toList());
        }
        this.lastExportSkippedData.set(mustFilter);
        if (spannerMetricData.isEmpty()) {
            return CompletableResultCode.ofSuccess();
        }
        try {
            spannerTimeSeries = SpannerCloudMonitoringExporterUtils.convertToSpannerTimeSeries(spannerMetricData);
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "Failed to convert spanner metric data to cloud monitoring timeseries.", e);
            return CompletableResultCode.ofFailure();
        }
        final ProjectName projectName = ProjectName.of((String)this.spannerProjectId);
        ApiFuture<List<Empty>> futureList = this.exportTimeSeriesInBatch(projectName, spannerTimeSeries);
        final CompletableResultCode spannerExportCode = new CompletableResultCode();
        ApiFutures.addCallback(futureList, (ApiFutureCallback)new ApiFutureCallback<List<Empty>>(){

            public void onFailure(Throwable throwable) {
                if (SpannerCloudMonitoringExporter.this.spannerExportFailureLogged.compareAndSet(false, true)) {
                    String msg = "createServiceTimeSeries request failed for spanner metrics.";
                    if (throwable instanceof PermissionDeniedException) {
                        msg = msg + String.format(" Need monitoring metric writer permission on project=%s. Follow https://cloud.google.com/spanner/docs/view-manage-client-side-metrics#access-client-side-metrics to set up permissions", projectName.getProject());
                    }
                    logger.log(Level.WARNING, msg, throwable);
                }
                spannerExportCode.fail();
            }

            public void onSuccess(List<Empty> empty) {
                SpannerCloudMonitoringExporter.this.spannerExportFailureLogged.set(false);
                spannerExportCode.succeed();
            }
        }, (Executor)MoreExecutors.directExecutor());
        return spannerExportCode;
    }

    private boolean shouldSkipMetricData(MetricData metricData) {
        return this.shouldSkipPointDataDueToProjectId(metricData.getResource());
    }

    private boolean shouldSkipPointDataDueToProjectId(Resource resource) {
        return !this.spannerProjectId.equals(SpannerCloudMonitoringExporterUtils.getProjectId(resource));
    }

    boolean lastExportSkippedData() {
        return this.lastExportSkippedData.get();
    }

    private ApiFuture<List<Empty>> exportTimeSeriesInBatch(ProjectName projectName, List<TimeSeries> timeSeries) {
        ArrayList<ApiFuture> batchResults = new ArrayList<ApiFuture>();
        for (List batch : Iterables.partition(timeSeries, (int)200)) {
            CreateTimeSeriesRequest req = CreateTimeSeriesRequest.newBuilder().setName(projectName.toString()).addAllTimeSeries((Iterable)batch).build();
            batchResults.add(this.client.createServiceTimeSeriesCallable().futureCall((Object)req));
        }
        return ApiFutures.allAsList(batchResults);
    }

    public CompletableResultCode flush() {
        return CompletableResultCode.ofSuccess();
    }

    public CompletableResultCode shutdown() {
        if (this.client.isShutdown()) {
            logger.log(Level.WARNING, "shutdown is called multiple times");
            return CompletableResultCode.ofSuccess();
        }
        CompletableResultCode shutdownResult = new CompletableResultCode();
        try {
            this.client.shutdown();
            shutdownResult.succeed();
        }
        catch (Throwable e) {
            logger.log(Level.WARNING, "failed to shutdown the monitoring client", e);
            shutdownResult.fail();
        }
        return shutdownResult;
    }

    public AggregationTemporality getAggregationTemporality(@Nonnull InstrumentType instrumentType) {
        return AggregationTemporality.CUMULATIVE;
    }
}

