/*
 * Decompiled with CFR 0.152.
 */
package com.github.victools.jsonschema.generator;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.BooleanNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.node.TextNode;
import com.github.victools.jsonschema.generator.CustomDefinition;
import com.github.victools.jsonschema.generator.JavaType;
import com.github.victools.jsonschema.generator.SchemaGeneratorConfig;
import com.github.victools.jsonschema.generator.TypeVariableContext;
import com.github.victools.jsonschema.generator.impl.AttributeCollector;
import com.github.victools.jsonschema.generator.impl.ReflectionTypeUtils;
import com.github.victools.jsonschema.generator.impl.SchemaGenerationContext;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaGenerator {
    private static final Logger logger = LoggerFactory.getLogger(SchemaGenerator.class);
    private final SchemaGeneratorConfig config;

    public SchemaGenerator(SchemaGeneratorConfig config) {
        this.config = config;
    }

    public JsonNode generateSchema(Class<?> mainTargetType) {
        ObjectNode definitionsNode;
        SchemaGenerationContext generationContext = new SchemaGenerationContext();
        JavaType mainType = new JavaType(mainTargetType, TypeVariableContext.EMPTY_SCOPE);
        this.traverseGenericType(mainType, null, false, generationContext);
        ObjectNode jsonSchemaResult = this.config.createObjectNode();
        if (this.config.shouldIncludeSchemaVersionIndicator()) {
            jsonSchemaResult.put("$schema", "http://json-schema.org/draft-07/schema#");
        }
        if ((definitionsNode = this.buildDefinitionsAndResolveReferences(mainType, generationContext)).size() > 0) {
            jsonSchemaResult.set("definitions", (JsonNode)definitionsNode);
        }
        ObjectNode mainSchemaNode = generationContext.getDefinition(mainType);
        jsonSchemaResult.setAll(mainSchemaNode);
        return jsonSchemaResult;
    }

    private void traverseGenericType(JavaType targetType, ObjectNode targetNode, boolean isNullable, SchemaGenerationContext generationContext) {
        if (generationContext.containsDefinition(targetType)) {
            logger.debug("adding reference to existing definition of {}", (Object)targetType);
            generationContext.addReference(targetType, targetNode, isNullable);
            return;
        }
        ObjectNode definition = ReflectionTypeUtils.isArrayType(targetType) ? this.traverseArrayType(targetType, targetNode, isNullable, generationContext) : this.traverseObjectType(targetType, targetNode, isNullable, generationContext);
        this.config.getTypeAttributeOverrides().forEach(override -> override.overrideTypeAttributes(definition, targetType, this.config));
    }

    private ObjectNode traverseArrayType(JavaType targetType, ObjectNode targetNode, boolean isNullable, SchemaGenerationContext generationContext) {
        ObjectNode definition;
        if (targetNode == null) {
            definition = this.config.createObjectNode();
            generationContext.putDefinition(targetType, definition);
        } else {
            definition = targetNode;
        }
        if (isNullable) {
            definition.set("type", (JsonNode)this.config.createArrayNode().add("array").add("null"));
        } else {
            definition.put("type", "array");
        }
        ObjectNode arrayItemTypeRef = this.config.createObjectNode();
        definition.set("items", (JsonNode)arrayItemTypeRef);
        JavaType itemType = ReflectionTypeUtils.getArrayComponentType(targetType);
        logger.debug("resolved array component type {}", (Object)itemType);
        this.traverseGenericType(itemType, arrayItemTypeRef, false, generationContext);
        return definition;
    }

    private ObjectNode traverseObjectType(JavaType targetType, ObjectNode targetNode, boolean isNullable, SchemaGenerationContext generationContext) {
        ObjectNode definition;
        CustomDefinition customDefinition = this.config.getCustomDefinition(targetType);
        if (customDefinition != null && customDefinition.isMeantToBeInline()) {
            if (targetNode == null) {
                logger.debug("storing configured custom inline type for {} as definition (since it is the main schema \"#\")", (Object)targetType);
                definition = customDefinition.getValue();
                generationContext.putDefinition(targetType, definition);
            } else {
                logger.debug("directly applying configured custom inline type for {}", (Object)targetType);
                targetNode.setAll(customDefinition.getValue());
                definition = targetNode;
            }
        } else {
            definition = this.config.createObjectNode();
            generationContext.putDefinition(targetType, definition);
            if (targetNode != null) {
                generationContext.addReference(targetType, targetNode, isNullable);
            }
            if (customDefinition == null) {
                logger.debug("generating definition for {}", (Object)targetType);
                definition.put("type", "object");
                TreeMap<String, JsonNode> targetFields = new TreeMap<String, JsonNode>();
                TreeMap<String, JsonNode> targetMethods = new TreeMap<String, JsonNode>();
                this.collectObjectProperties(targetType, targetFields, targetMethods, generationContext);
                if (!targetFields.isEmpty() || !targetMethods.isEmpty()) {
                    ObjectNode propertiesNode = this.config.createObjectNode();
                    propertiesNode.setAll(targetFields);
                    propertiesNode.setAll(targetMethods);
                    definition.set("properties", (JsonNode)propertiesNode);
                }
            } else {
                logger.debug("applying configured custom definition for {}", (Object)targetType);
                definition.setAll(customDefinition.getValue());
            }
        }
        return definition;
    }

    private void collectObjectProperties(JavaType targetType, Map<String, JsonNode> targetFields, Map<String, JsonNode> targetMethods, SchemaGenerationContext generationContext) {
        TypeVariableContext targetTypeVariables = TypeVariableContext.forType(targetType);
        Class<?> currentTargetClass = ReflectionTypeUtils.getRawType(targetType.getResolvedType());
        logger.debug("iterating over declared fields from {}", currentTargetClass);
        Stream.of(currentTargetClass.getDeclaredFields()).filter(declaredField -> !declaredField.isSynthetic() && !this.config.shouldIgnore((Field)declaredField)).forEach(field -> this.populateField((Field)field, targetFields, targetTypeVariables, generationContext));
        logger.debug("iterating over declared public methods from {}", currentTargetClass);
        Stream.of(currentTargetClass.getDeclaredMethods()).filter(declaredMethod -> (declaredMethod.getModifiers() & 1) == 1 && !declaredMethod.isSynthetic()).filter(declaredMethod -> !currentTargetClass.isInterface() || declaredMethod.isDefault()).filter(declaredMethod -> !this.config.shouldIgnore((Method)declaredMethod)).forEach(method -> this.populateMethod((Method)method, targetMethods, targetTypeVariables, generationContext));
        JavaType superType = new JavaType(currentTargetClass.getGenericSuperclass(), targetTypeVariables);
        if (superType.getResolvedType() != null && superType.getResolvedType() != Object.class) {
            this.collectObjectProperties(superType, targetFields, targetMethods, generationContext);
        }
        Stream.of(currentTargetClass.getGenericInterfaces()).map(superInterface -> new JavaType((Type)superInterface, targetTypeVariables)).forEach(superInterface -> this.collectObjectProperties((JavaType)superInterface, targetFields, targetMethods, generationContext));
    }

    private void populateField(Field field, Map<String, JsonNode> parentProperties, TypeVariableContext typeVariables, SchemaGenerationContext generationContext) {
        String defaultName = field.getName();
        String propertyName = Optional.ofNullable(this.config.resolvePropertyNameOverride(field, defaultName)).orElse(defaultName);
        if (parentProperties.containsKey(propertyName)) {
            logger.debug("ignoring overridden {}.{}", field.getDeclaringClass(), (Object)defaultName);
            return;
        }
        ObjectNode subSchema = this.config.createObjectNode();
        parentProperties.put(propertyName, (JsonNode)subSchema);
        JavaType fieldType = typeVariables.resolveGenericTypePlaceholder(field.getGenericType());
        boolean isNullable = !field.isEnumConstant() && this.config.isNullable(field, fieldType);
        fieldType = Optional.ofNullable(this.config.resolveTargetTypeOverride(field, fieldType)).orElse(fieldType);
        ObjectNode fieldAttributes = AttributeCollector.collectFieldAttributes(field, fieldType, this.config);
        this.populateSchema(fieldType, subSchema, isNullable, fieldAttributes, generationContext);
    }

    private void populateMethod(Method method, Map<String, JsonNode> parentProperties, TypeVariableContext parentTypeVariables, SchemaGenerationContext generationContext) {
        TypeVariableContext extendedTypeVariables = TypeVariableContext.forMethod(method, parentTypeVariables);
        JavaType returnValueType = extendedTypeVariables.resolveGenericTypePlaceholder(method.getGenericReturnType());
        boolean isNullable = this.config.isNullable(method, returnValueType);
        returnValueType = Optional.ofNullable(this.config.resolveTargetTypeOverride(method, returnValueType)).orElse(returnValueType);
        String defaultName = method.getName() + Stream.of(method.getGenericParameterTypes()).map(parameter -> extendedTypeVariables.resolveGenericTypePlaceholder((Type)parameter).toString()).collect(Collectors.joining(", ", "(", ")"));
        String propertyName = Optional.ofNullable(this.config.resolvePropertyNameOverride(method, defaultName)).orElse(defaultName);
        if (parentProperties.containsKey(propertyName)) {
            logger.debug("ignoring overridden {}.{}", method.getDeclaringClass(), (Object)defaultName);
            return;
        }
        if (returnValueType.getResolvedType() == Void.TYPE || returnValueType.getResolvedType() == Void.class) {
            parentProperties.put(propertyName, (JsonNode)BooleanNode.FALSE);
        } else {
            ObjectNode subSchema = this.config.createObjectNode();
            parentProperties.put(propertyName, (JsonNode)subSchema);
            ObjectNode methodAttributes = AttributeCollector.collectMethodAttributes(method, returnValueType, this.config);
            this.populateSchema(returnValueType, subSchema, isNullable, methodAttributes, generationContext);
        }
    }

    private void populateSchema(JavaType javaType, ObjectNode targetNode, boolean isNullable, ObjectNode collectedAttributes, SchemaGenerationContext generationContext) {
        ObjectNode referenceContainer;
        CustomDefinition customDefinition = this.config.getCustomDefinition(javaType);
        if (collectedAttributes == null || collectedAttributes.size() == 0 || customDefinition != null && customDefinition.isMeantToBeInline() || ReflectionTypeUtils.isArrayType(javaType)) {
            referenceContainer = targetNode;
        } else {
            referenceContainer = this.config.createObjectNode();
            targetNode.set("allOf", (JsonNode)this.config.createArrayNode().add((JsonNode)referenceContainer).add((JsonNode)collectedAttributes));
        }
        if (customDefinition != null && customDefinition.isMeantToBeInline()) {
            referenceContainer.setAll(customDefinition.getValue());
            if (collectedAttributes != null && collectedAttributes.size() > 0) {
                referenceContainer.setAll(collectedAttributes);
            }
            if (isNullable) {
                this.makeNullable(referenceContainer);
            }
        } else {
            try {
                this.traverseGenericType(javaType, referenceContainer, isNullable, generationContext);
            }
            catch (UnsupportedOperationException ex) {
                logger.warn("Skipping type definition due to error", (Throwable)ex);
            }
        }
    }

    private void makeNullable(ObjectNode node) {
        if (node.has("$ref") || node.has("allOf") || node.has("anyOf") || node.has("oneOf")) {
            ObjectNode nullSchema = this.config.createObjectNode().put("type", "null");
            ArrayNode oneOf = this.config.createArrayNode().add((JsonNode)nullSchema).add(this.config.createObjectNode().setAll(node));
            node.removeAll();
            node.set("oneOf", (JsonNode)oneOf);
        } else {
            JsonNode fixedJsonSchemaType = node.get("type");
            if (fixedJsonSchemaType instanceof ArrayNode) {
                ArrayNode arrayOfTypes = (ArrayNode)fixedJsonSchemaType;
                boolean alreadyContainsNull = false;
                for (JsonNode arrayEntry : arrayOfTypes) {
                    alreadyContainsNull = alreadyContainsNull || "null".equals(arrayEntry.textValue());
                }
                if (!alreadyContainsNull) {
                    arrayOfTypes.add("null");
                }
            } else if (fixedJsonSchemaType instanceof TextNode && !"null".equals(fixedJsonSchemaType.textValue())) {
                node.replace("type", (JsonNode)this.config.createArrayNode().add(fixedJsonSchemaType).add("null"));
            }
        }
    }

    private ObjectNode buildDefinitionsAndResolveReferences(JavaType mainSchemaTarget, SchemaGenerationContext generationContext) {
        Map aliases = generationContext.getDefinedTypes().stream().collect(Collectors.groupingBy(type -> type.toString(), TreeMap::new, Collectors.toList()));
        ObjectNode definitionsNode = this.config.createObjectNode();
        boolean createDefinitionsForAll = this.config.shouldCreateDefinitionsForAllObjects();
        for (Map.Entry aliasEntry : aliases.entrySet()) {
            String referenceKey;
            boolean referenceInline;
            List types = (List)aliasEntry.getValue();
            List<ObjectNode> referencingNodes = types.stream().flatMap(type -> generationContext.getReferences((JavaType)type).stream()).collect(Collectors.toList());
            List<ObjectNode> nullableReferences = types.stream().flatMap(type -> generationContext.getNullableReferences((JavaType)type).stream()).collect(Collectors.toList());
            String alias = (String)aliasEntry.getKey();
            boolean bl = referenceInline = !types.contains(mainSchemaTarget) && (referencingNodes.isEmpty() || !createDefinitionsForAll && referencingNodes.size() + nullableReferences.size() < 2);
            if (referenceInline) {
                referencingNodes.forEach(referenceNode -> referenceNode.setAll(generationContext.getDefinition((JavaType)types.get(0))));
                referenceKey = null;
            } else {
                if (types.contains(mainSchemaTarget)) {
                    referenceKey = "#";
                } else {
                    definitionsNode.set(alias, (JsonNode)generationContext.getDefinition((JavaType)types.get(0)));
                    referenceKey = "#/definitions/" + alias;
                }
                referencingNodes.forEach(referenceNode -> referenceNode.put("$ref", referenceKey));
            }
            if (nullableReferences.isEmpty()) continue;
            ObjectNode definition = referenceInline ? generationContext.getDefinition((JavaType)types.get(0)) : this.config.createObjectNode().put("$ref", referenceKey);
            this.makeNullable(definition);
            if (createDefinitionsForAll || nullableReferences.size() > 1) {
                String nullableAlias = alias + " (nullable)";
                String nullableReferenceKey = "#/definitions/" + nullableAlias;
                definitionsNode.set(nullableAlias, (JsonNode)definition);
                nullableReferences.forEach(referenceNode -> referenceNode.put("$ref", nullableReferenceKey));
                continue;
            }
            nullableReferences.forEach(referenceNode -> referenceNode.setAll(definition));
        }
        return definitionsNode;
    }
}

