/*
 * Decompiled with CFR 0.152.
 */
package org.tron.core.net.peer;

import com.google.common.collect.Lists;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;
import org.tron.common.prometheus.Metrics;
import org.tron.core.net.peer.PeerConnection;
import org.tron.p2p.connection.Channel;

public class PeerManager {
    private static final Logger logger = LoggerFactory.getLogger((String)"net");
    private static List<PeerConnection> peers = Collections.synchronizedList(new ArrayList());
    private static AtomicInteger passivePeersCount = new AtomicInteger(0);
    private static AtomicInteger activePeersCount = new AtomicInteger(0);
    private static ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private static long DISCONNECTION_TIME_OUT = 60000L;

    public static void init() {
        executor.scheduleWithFixedDelay(() -> {
            try {
                PeerManager.check();
                PeerManager.logPeerStats();
            }
            catch (Throwable t) {
                logger.error("Exception in peer manager", t);
            }
        }, 30L, 10L, TimeUnit.SECONDS);
    }

    public static void close() {
        try {
            for (PeerConnection p : new ArrayList<PeerConnection>(peers)) {
                if (p.isDisconnect()) continue;
                p.getChannel().close();
            }
            executor.shutdownNow();
        }
        catch (Exception e) {
            logger.error("Peer manager shutdown failed", (Throwable)e);
        }
    }

    public static synchronized PeerConnection add(ApplicationContext ctx, Channel channel) {
        PeerConnection peerConnection = PeerManager.getPeerConnection(channel);
        if (peerConnection != null) {
            return null;
        }
        peerConnection = (PeerConnection)ctx.getBean(PeerConnection.class);
        peerConnection.setChannel(channel);
        peers.add(peerConnection);
        if (channel.isActive()) {
            activePeersCount.incrementAndGet();
        } else {
            passivePeersCount.incrementAndGet();
        }
        return peerConnection;
    }

    public static synchronized PeerConnection remove(Channel channel) {
        PeerConnection peerConnection = PeerManager.getPeerConnection(channel);
        if (peerConnection == null) {
            return null;
        }
        PeerManager.remove(peerConnection);
        return peerConnection;
    }

    private static void remove(PeerConnection peerConnection) {
        peers.remove(peerConnection);
        if (peerConnection.getChannel().isActive()) {
            activePeersCount.decrementAndGet();
        } else {
            passivePeersCount.decrementAndGet();
        }
    }

    public static synchronized void sortPeers() {
        peers.sort(Comparator.comparingDouble(c -> c.getChannel().getAvgLatency()));
    }

    public static PeerConnection getPeerConnection(Channel channel) {
        for (PeerConnection peer : new ArrayList<PeerConnection>(peers)) {
            if (!peer.getChannel().equals((Object)channel)) continue;
            return peer;
        }
        return null;
    }

    public static List<PeerConnection> getPeers() {
        ArrayList peers = Lists.newArrayList();
        for (PeerConnection peer : new ArrayList<PeerConnection>(PeerManager.peers)) {
            if (peer.isDisconnect()) continue;
            peers.add(peer);
        }
        return peers;
    }

    private static void check() {
        long now = System.currentTimeMillis();
        for (PeerConnection peer : new ArrayList<PeerConnection>(peers)) {
            long disconnectTime = peer.getChannel().getDisconnectTime();
            if (disconnectTime == 0L || now - disconnectTime <= DISCONNECTION_TIME_OUT) continue;
            logger.warn("Notify disconnect peer {}.", (Object)peer.getInetSocketAddress());
            peers.remove(peer);
            if (peer.getChannel().isActive()) {
                activePeersCount.decrementAndGet();
            } else {
                passivePeersCount.decrementAndGet();
            }
            peer.onDisconnect();
        }
    }

    private static synchronized void logPeerStats() {
        String str = String.format("\n\n============ Peer stats: all %d, active %d, passive %d\n\n", peers.size(), activePeersCount.get(), passivePeersCount.get());
        PeerManager.metric(peers.size(), "all");
        PeerManager.metric(activePeersCount.get(), "active");
        PeerManager.metric(passivePeersCount.get(), "passive");
        StringBuilder sb = new StringBuilder(str);
        int valid = 0;
        for (PeerConnection peer : new ArrayList<PeerConnection>(peers)) {
            sb.append(peer.log());
            sb.append("\n");
            if (peer.isNeedSyncFromUs() || peer.isNeedSyncFromPeer()) continue;
            ++valid;
        }
        PeerManager.metric(valid, "valid");
        logger.info(sb.toString());
    }

    private static void metric(double amt, String peerType) {
        Metrics.gaugeSet((String)"tron:peers", (double)amt, (String[])new String[]{peerType});
    }

    public static AtomicInteger getPassivePeersCount() {
        return passivePeersCount;
    }

    public static AtomicInteger getActivePeersCount() {
        return activePeersCount;
    }
}

