/*
 * Decompiled with CFR 0.152.
 */
package org.tron.core.service;

import com.google.protobuf.ByteString;
import java.math.BigInteger;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.collections4.CollectionUtils;
import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.tron.common.utils.Pair;
import org.tron.common.utils.StringUtil;
import org.tron.core.capsule.AccountCapsule;
import org.tron.core.capsule.WitnessCapsule;
import org.tron.core.exception.BalanceInsufficientException;
import org.tron.core.service.RewardViCalService;
import org.tron.core.store.AccountStore;
import org.tron.core.store.DelegationStore;
import org.tron.core.store.DynamicPropertiesStore;
import org.tron.core.store.WitnessStore;

@Component
public class MortgageService {
    private static final Logger logger = LoggerFactory.getLogger((String)"mortgage");
    private WitnessStore witnessStore;
    private DelegationStore delegationStore;
    private DynamicPropertiesStore dynamicPropertiesStore;
    private AccountStore accountStore;
    @Autowired
    private RewardViCalService rewardViCalService;

    public void initStore(WitnessStore witnessStore, DelegationStore delegationStore, DynamicPropertiesStore dynamicPropertiesStore, AccountStore accountStore) {
        this.witnessStore = witnessStore;
        this.delegationStore = delegationStore;
        this.dynamicPropertiesStore = dynamicPropertiesStore;
        this.accountStore = accountStore;
    }

    public void payStandbyWitness() {
        List<WitnessCapsule> witnessStandbys = this.witnessStore.getWitnessStandby();
        long voteSum = witnessStandbys.stream().mapToLong(WitnessCapsule::getVoteCount).sum();
        if (voteSum < 1L) {
            return;
        }
        long totalPay = this.dynamicPropertiesStore.getWitness127PayPerBlock();
        double eachVotePay = (double)totalPay / (double)voteSum;
        for (WitnessCapsule w : witnessStandbys) {
            long pay = (long)((double)w.getVoteCount() * eachVotePay);
            this.payReward(w.getAddress().toByteArray(), pay);
            logger.debug("Pay {} stand reward {}.", (Object)Hex.toHexString((byte[])w.getAddress().toByteArray()), (Object)pay);
        }
    }

    public void payBlockReward(byte[] witnessAddress, long value) {
        logger.debug("Pay {} block reward {}.", (Object)Hex.toHexString((byte[])witnessAddress), (Object)value);
        this.payReward(witnessAddress, value);
    }

    public void payTransactionFeeReward(byte[] witnessAddress, long value) {
        logger.debug("Pay {} transaction fee reward {}.", (Object)Hex.toHexString((byte[])witnessAddress), (Object)value);
        this.payReward(witnessAddress, value);
    }

    private void payReward(byte[] witnessAddress, long value) {
        long cycle = this.dynamicPropertiesStore.getCurrentCycleNumber();
        int brokerage = this.delegationStore.getBrokerage(cycle, witnessAddress);
        double brokerageRate = (double)brokerage / 100.0;
        long brokerageAmount = (long)(brokerageRate * (double)value);
        this.delegationStore.addReward(cycle, witnessAddress, value -= brokerageAmount);
        this.adjustAllowance(witnessAddress, brokerageAmount);
    }

    public void withdrawReward(byte[] address) {
        AccountCapsule account;
        if (!this.dynamicPropertiesStore.allowChangeDelegation()) {
            return;
        }
        AccountCapsule accountCapsule = this.accountStore.get(address);
        long beginCycle = this.delegationStore.getBeginCycle(address);
        long endCycle = this.delegationStore.getEndCycle(address);
        long currentCycle = this.dynamicPropertiesStore.getCurrentCycleNumber();
        long reward = 0L;
        if (beginCycle > currentCycle || accountCapsule == null) {
            return;
        }
        if (beginCycle == currentCycle && (account = this.delegationStore.getAccountVote(beginCycle, address)) != null) {
            return;
        }
        if (beginCycle + 1L == endCycle && beginCycle < currentCycle) {
            account = this.delegationStore.getAccountVote(beginCycle, address);
            if (account != null) {
                reward = this.computeReward(beginCycle, endCycle, account);
                this.adjustAllowance(address, reward);
                reward = 0L;
                logger.info("Latest cycle reward {}, {}.", (Object)beginCycle, account.getVotesList());
            }
            ++beginCycle;
        }
        endCycle = currentCycle;
        if (CollectionUtils.isEmpty(accountCapsule.getVotesList())) {
            this.delegationStore.setBeginCycle(address, endCycle + 1L);
            return;
        }
        if (beginCycle < endCycle) {
            this.adjustAllowance(address, reward += this.computeReward(beginCycle, endCycle, accountCapsule));
        }
        this.delegationStore.setBeginCycle(address, endCycle);
        this.delegationStore.setEndCycle(address, endCycle + 1L);
        this.delegationStore.setAccountVote(endCycle, address, accountCapsule);
        logger.info("Adjust {} allowance {}, now currentCycle {}, beginCycle {}, endCycle {}, account vote {}.", new Object[]{Hex.toHexString((byte[])address), reward, currentCycle, beginCycle, endCycle, accountCapsule.getVotesList()});
    }

    public long queryReward(byte[] address) {
        if (!this.dynamicPropertiesStore.allowChangeDelegation()) {
            return 0L;
        }
        AccountCapsule accountCapsule = this.accountStore.get(address);
        long beginCycle = this.delegationStore.getBeginCycle(address);
        long endCycle = this.delegationStore.getEndCycle(address);
        long currentCycle = this.dynamicPropertiesStore.getCurrentCycleNumber();
        long reward = 0L;
        if (accountCapsule == null) {
            return 0L;
        }
        if (beginCycle > currentCycle) {
            return accountCapsule.getAllowance();
        }
        if (beginCycle + 1L == endCycle && beginCycle < currentCycle) {
            AccountCapsule account = this.delegationStore.getAccountVote(beginCycle, address);
            if (account != null) {
                reward = this.computeReward(beginCycle, endCycle, account);
            }
            ++beginCycle;
        }
        endCycle = currentCycle;
        if (CollectionUtils.isEmpty(accountCapsule.getVotesList())) {
            return reward + accountCapsule.getAllowance();
        }
        if (beginCycle < endCycle) {
            reward += this.computeReward(beginCycle, endCycle, accountCapsule);
        }
        return reward + accountCapsule.getAllowance();
    }

    private long computeReward(long cycle, List<Pair<byte[], Long>> votes) {
        long reward = 0L;
        for (Pair<byte[], Long> vote : votes) {
            long totalVote;
            byte[] srAddress = (byte[])vote.getKey();
            long totalReward = this.delegationStore.getReward(cycle, srAddress);
            if (totalReward <= 0L || (totalVote = this.delegationStore.getWitnessVote(cycle, srAddress)) == -1L || totalVote == 0L) continue;
            long userVote = (Long)vote.getValue();
            double voteRate = (double)userVote / (double)totalVote;
            reward = (long)((double)reward + voteRate * (double)totalReward);
        }
        return reward;
    }

    private long computeReward(long beginCycle, long endCycle, AccountCapsule accountCapsule) {
        if (beginCycle >= endCycle) {
            return 0L;
        }
        long reward = 0L;
        long newAlgorithmCycle = this.dynamicPropertiesStore.getNewRewardAlgorithmEffectiveCycle();
        List<Pair<byte[], Long>> srAddresses = accountCapsule.getVotesList().stream().map(vote -> new Pair((Object)vote.getVoteAddress().toByteArray(), (Object)vote.getVoteCount())).collect(Collectors.toList());
        if (beginCycle < newAlgorithmCycle) {
            long oldEndCycle = Math.min(endCycle, newAlgorithmCycle);
            reward = this.getOldReward(beginCycle, oldEndCycle, srAddresses);
            beginCycle = oldEndCycle;
        }
        if (beginCycle < endCycle) {
            for (Pair pair : srAddresses) {
                byte[] srAddress = (byte[])pair.getKey();
                BigInteger beginVi = this.delegationStore.getWitnessVi(beginCycle - 1L, srAddress);
                BigInteger endVi = this.delegationStore.getWitnessVi(endCycle - 1L, srAddress);
                BigInteger deltaVi = endVi.subtract(beginVi);
                if (deltaVi.signum() <= 0) continue;
                long userVote = (Long)pair.getValue();
                reward += deltaVi.multiply(BigInteger.valueOf(userVote)).divide(DelegationStore.DECIMAL_OF_VI_REWARD).longValue();
            }
        }
        return reward;
    }

    public WitnessCapsule getWitnessByAddress(ByteString address) {
        return this.witnessStore.get(address.toByteArray());
    }

    public void adjustAllowance(byte[] address, long amount) {
        try {
            if (amount <= 0L) {
                return;
            }
            this.adjustAllowance(this.accountStore, address, amount);
        }
        catch (BalanceInsufficientException e) {
            logger.error("WithdrawReward error: {}.", (Object)e.getMessage());
        }
    }

    public void adjustAllowance(AccountStore accountStore, byte[] accountAddress, long amount) throws BalanceInsufficientException {
        AccountCapsule account = (AccountCapsule)accountStore.getUnchecked(accountAddress);
        long allowance = account.getAllowance();
        if (amount == 0L) {
            return;
        }
        if (amount < 0L && allowance < -amount) {
            throw new BalanceInsufficientException(String.format("%s insufficient balance, amount: %d, allowance: %d", StringUtil.createReadableString((byte[])accountAddress), amount, allowance));
        }
        account.setAllowance(allowance + amount);
        accountStore.put(account.createDbKey(), account);
    }

    private void sortWitness(List<ByteString> list) {
        list.sort(Comparator.comparingLong(b -> this.getWitnessByAddress((ByteString)b).getVoteCount()).reversed().thenComparing(Comparator.comparingInt(ByteString::hashCode).reversed()));
    }

    private long getOldReward(long begin, long end, List<Pair<byte[], Long>> votes) {
        if (this.dynamicPropertiesStore.allowOldRewardOpt()) {
            return this.rewardViCalService.getNewRewardAlgorithmReward(begin, end, votes);
        }
        long reward = 0L;
        for (long cycle = begin; cycle < end; ++cycle) {
            reward += this.computeReward(cycle, votes);
        }
        return reward;
    }

    public void setWitnessStore(WitnessStore witnessStore) {
        this.witnessStore = witnessStore;
    }

    public void setDelegationStore(DelegationStore delegationStore) {
        this.delegationStore = delegationStore;
    }

    public DelegationStore getDelegationStore() {
        return this.delegationStore;
    }

    public void setDynamicPropertiesStore(DynamicPropertiesStore dynamicPropertiesStore) {
        this.dynamicPropertiesStore = dynamicPropertiesStore;
    }

    public void setAccountStore(AccountStore accountStore) {
        this.accountStore = accountStore;
    }
}

