/*
 * Decompiled with CFR 0.152.
 */
package org.tron.core.vm.nativecontract;

import com.google.protobuf.ByteString;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.tron.common.utils.FastByteComparisons;
import org.tron.core.capsule.AccountCapsule;
import org.tron.core.capsule.DelegatedResourceCapsule;
import org.tron.core.exception.ContractValidateException;
import org.tron.core.store.DynamicPropertiesStore;
import org.tron.core.vm.nativecontract.param.FreezeBalanceParam;
import org.tron.core.vm.repository.Repository;
import org.tron.protos.Protocol;

public class FreezeBalanceProcessor {
    private static final Logger logger = LoggerFactory.getLogger((String)"VMProcessor");

    public void validate(FreezeBalanceParam param, Repository repo) throws ContractValidateException {
        if (repo == null) {
            throw new ContractValidateException("No account store or dynamic store!");
        }
        byte[] ownerAddress = param.getOwnerAddress();
        AccountCapsule ownerCapsule = repo.getAccount(ownerAddress);
        long frozenBalance = param.getFrozenBalance();
        if (frozenBalance <= 0L) {
            throw new ContractValidateException("FrozenBalance must be positive");
        }
        if (frozenBalance < 1000000L) {
            throw new ContractValidateException("FrozenBalance must be greater than or equal to 1 TRX");
        }
        if (frozenBalance > ownerCapsule.getBalance()) {
            throw new ContractValidateException("FrozenBalance must be less than or equal to accountBalance");
        }
        int frozenCount = ownerCapsule.getFrozenCount();
        if (frozenCount != 0 && frozenCount != 1) {
            throw new ContractValidateException("FrozenCount must be 0 or 1");
        }
        switch (param.getResourceType()) {
            case BANDWIDTH: 
            case ENERGY: {
                break;
            }
            default: {
                throw new ContractValidateException("Unknown ResourceCode, valid ResourceCode[BANDWIDTH\u3001ENERGY]");
            }
        }
        byte[] receiverAddress = param.getReceiverAddress();
        if (!FastByteComparisons.isEqual((byte[])ownerAddress, (byte[])receiverAddress)) {
            param.setDelegating(true);
            AccountCapsule receiverCapsule = repo.getAccount(receiverAddress);
            if (receiverCapsule == null) {
                receiverCapsule = repo.createNormalAccount(receiverAddress);
            }
            if (receiverCapsule.getType() == Protocol.AccountType.Contract) {
                throw new ContractValidateException("Do not allow delegate resources to contract addresses");
            }
        }
    }

    public void execute(FreezeBalanceParam param, Repository repo) {
        DynamicPropertiesStore dynamicStore = repo.getDynamicPropertiesStore();
        long nowInMs = dynamicStore.getLatestBlockHeaderTimestamp();
        long expireTime = nowInMs + param.getFrozenDuration() * 86400000L;
        byte[] ownerAddress = param.getOwnerAddress();
        byte[] receiverAddress = param.getReceiverAddress();
        long frozenBalance = param.getFrozenBalance();
        AccountCapsule accountCapsule = repo.getAccount(ownerAddress);
        if (param.isDelegating()) {
            switch (param.getResourceType()) {
                case BANDWIDTH: {
                    this.delegateResource(ownerAddress, receiverAddress, frozenBalance, expireTime, true, repo);
                    accountCapsule.addDelegatedFrozenBalanceForBandwidth(frozenBalance);
                    break;
                }
                case ENERGY: {
                    this.delegateResource(ownerAddress, receiverAddress, frozenBalance, expireTime, false, repo);
                    accountCapsule.addDelegatedFrozenBalanceForEnergy(frozenBalance);
                    break;
                }
                default: {
                    logger.debug("Resource Code Error.");
                    break;
                }
            }
        } else {
            switch (param.getResourceType()) {
                case BANDWIDTH: {
                    accountCapsule.setFrozenForBandwidth(frozenBalance + accountCapsule.getFrozenBalance(), expireTime);
                    break;
                }
                case ENERGY: {
                    accountCapsule.setFrozenForEnergy(frozenBalance + accountCapsule.getAccountResource().getFrozenBalanceForEnergy().getFrozenBalance(), expireTime);
                    break;
                }
                default: {
                    logger.debug("Resource Code Error.");
                }
            }
        }
        switch (param.getResourceType()) {
            case BANDWIDTH: {
                repo.addTotalNetWeight(frozenBalance / 1000000L);
                break;
            }
            case ENERGY: {
                repo.addTotalEnergyWeight(frozenBalance / 1000000L);
                break;
            }
        }
        long newBalance = accountCapsule.getBalance() - frozenBalance;
        accountCapsule.setBalance(newBalance);
        repo.updateAccount(accountCapsule.createDbKey(), accountCapsule);
    }

    private void delegateResource(byte[] ownerAddress, byte[] receiverAddress, long frozenBalance, long expireTime, boolean isBandwidth, Repository repo) {
        byte[] key = DelegatedResourceCapsule.createDbKey((byte[])ownerAddress, (byte[])receiverAddress);
        DelegatedResourceCapsule delegatedResourceCapsule = repo.getDelegatedResource(key);
        if (delegatedResourceCapsule == null) {
            delegatedResourceCapsule = new DelegatedResourceCapsule(ByteString.copyFrom((byte[])ownerAddress), ByteString.copyFrom((byte[])receiverAddress));
        }
        if (isBandwidth) {
            delegatedResourceCapsule.addFrozenBalanceForBandwidth(frozenBalance, expireTime);
        } else {
            delegatedResourceCapsule.addFrozenBalanceForEnergy(frozenBalance, expireTime);
        }
        repo.updateDelegatedResource(key, delegatedResourceCapsule);
        AccountCapsule receiverCapsule = repo.getAccount(receiverAddress);
        if (isBandwidth) {
            receiverCapsule.addAcquiredDelegatedFrozenBalanceForBandwidth(frozenBalance);
        } else {
            receiverCapsule.addAcquiredDelegatedFrozenBalanceForEnergy(frozenBalance);
        }
        repo.updateAccount(receiverCapsule.createDbKey(), receiverCapsule);
    }
}

