package com.proximities.sdk;

import android.app.Activity;
import android.app.Service;
import android.bluetooth.BluetoothAdapter;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.os.IBinder;
import android.util.Log;

import com.proximities.sdk.corekit.CoreKitInstance;
import com.proximities.sdk.corekit.interfaces.ActivityCallbacksInterface;
import com.proximities.sdk.interfaces.EnableLocAndBeaconInterface;
import com.proximities.sdk.util.ProximitiesConstants;
import com.proximities.sdk.util.ProximitiesPrefs;

import org.altbeacon.beacon.BeaconConsumer;

/**
 * Created by william
 *
 * This service is responsible for establishing and maintaining a
 * permanent connection between the device and user's position plus
 * beacon manager
 */
public class ProximitiesService extends Service implements BeaconConsumer, ActivityCallbacksInterface, EnableLocAndBeaconInterface{

    private CoreKitInstance coreKitInstance;
    private PSManager mPSManager;
    private ProximitiesBeaconManager proximitiesBeaconManager;
    private ProximitiesLocationManager proximitiesLocationManager;
    private ConnectionChangesReceiver connectionChangesReceiver = new ConnectionChangesReceiver();

    @Override
    public void onCreate() {
        super.onCreate();
        registerReceiver(mReceiver, new IntentFilter(BluetoothAdapter.ACTION_STATE_CHANGED));
        registerReceiver(connectionChangesReceiver, new IntentFilter(ConnectivityManager.CONNECTIVITY_ACTION));
    }

    @Override
    public void onActivityResumed(Activity activity) {
        mPSManager.handleCampaignsActivities(activity);
    }

    @Override
    public int onStartCommand(Intent intent, int flags, int startId) {
        ProximitiesPrefs.writeServiceIsStopped(getApplicationContext(), false);
        if(mPSManager == null){
            mPSManager = PSManager.getInstance();
        }
        if(coreKitInstance == null){
            coreKitInstance = CoreKitInstance.getSharedInstance();
            coreKitInstance.attachActivityCallbackInterface(this);
            ProximitiesConfig.setManagersCallback(this);
        }
        if(intent != null) {
            // when the service is started using startService()
            if(intent.getBooleanExtra(ProximitiesConstants.BOOT_INTENT_EXTRA, false)){
                startLocationDetection();
                startBeaconManager();
            } else {
                ProximitiesPrefs.writeStartLocationManager(getApplicationContext(), false);
                ProximitiesPrefs.writeStartBeaconManager(getApplicationContext(), false);
            }
        } else {
            // when the service is recreated (START_STICKY)
            startLocationDetection();
            startBeaconManager();
            ProximitiesConfig.getInstance().setServiceStarted(true);
        }
        if(ProximitiesConfig.getOnSweepinConnectServiceReady() != null){
            ProximitiesConfig.getOnSweepinConnectServiceReady().onSweepinConnectServiceReady();
        }
        return START_STICKY;
    }

    @Override
    public void enableLocationManager() {
        startLocationDetection();
    }

    @Override
    public void enableBeaconManager() {
        startBeaconManager();
    }

    private void startLocationDetection(){
        if(ProximitiesPrefs.readStartLocationManager(getApplicationContext())){
            proximitiesLocationManager = ProximitiesLocationManager.getInstance();
            proximitiesLocationManager.initLocationManager(getApplication());
        }
    }

    private void startBeaconManager(){
        if(ProximitiesPrefs.readStartBeaconManager(getApplicationContext())
                && BluetoothAdapter.getDefaultAdapter() != null && BluetoothAdapter.getDefaultAdapter().isEnabled()) {
            proximitiesBeaconManager = ProximitiesBeaconManager.getInstance();
            proximitiesBeaconManager.initBeaconManager(getApplicationContext(), this);
        }
    }

    @Override
    public void onBeaconServiceConnect() {
        proximitiesBeaconManager.startServicesWhenReady();
    }

    @Override
    public void onDestroy() {
        if(proximitiesBeaconManager != null){
            proximitiesBeaconManager.unbindBeaconManager(this);
        }
        unregisterReceiver(mReceiver);
        unregisterReceiver(connectionChangesReceiver);
        super.onDestroy();
    }

    @Override
    public IBinder onBind(Intent intent) {
        return null;
    }

    /*******************************************************************************************************/
    /*********************************** RECEIVERS ********************************************************/
    /*******************************************************************************************************/


    private final BroadcastReceiver mReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            final String action = intent.getAction();

            if (action.equals(BluetoothAdapter.ACTION_STATE_CHANGED)) {
                final int state = intent.getIntExtra(BluetoothAdapter.EXTRA_STATE,
                        BluetoothAdapter.ERROR);
                switch (state) {
                    case BluetoothAdapter.STATE_ON:
                        startBeaconManager();
                        break;
                    case BluetoothAdapter.STATE_OFF:
                        if(proximitiesBeaconManager != null){
                            proximitiesBeaconManager.unbindBeaconManager(ProximitiesService.this);
                            proximitiesBeaconManager = null;
                        }
                        break;
                }
            }
        }
    };

    private class ConnectionChangesReceiver extends BroadcastReceiver {

        private final String CONNECTIVITY_CHANGES_ACTION = "android.net.conn.CONNECTIVITY_CHANGE";

        public ConnectionChangesReceiver(){

        }

        @Override
        public void onReceive(Context context, Intent intent) {
            if (intent.getAction().equalsIgnoreCase(CONNECTIVITY_CHANGES_ACTION)){
                ConnectivityManager cm = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
                NetworkInfo activeNetwork = cm.getActiveNetworkInfo();
                boolean isConnected = activeNetwork != null && activeNetwork.isConnectedOrConnecting();
                if(isConnected) {
                    startLocationDetection();
                }
            }
        }
    }

}
