/*
 * Decompiled with CFR 0.152.
 */
package oshi.software.os.linux;

import com.sun.jna.Memory;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.jna.platform.linux.Libc;
import oshi.software.common.AbstractOperatingSystem;
import oshi.software.os.FileSystem;
import oshi.software.os.NetworkParams;
import oshi.software.os.OSProcess;
import oshi.software.os.OSUser;
import oshi.software.os.OperatingSystem;
import oshi.software.os.linux.LinuxFileSystem;
import oshi.software.os.linux.LinuxNetworkParams;
import oshi.software.os.linux.LinuxOSVersionInfoEx;
import oshi.software.os.linux.LinuxUserGroupInfo;
import oshi.util.ExecutingCommand;
import oshi.util.FileUtil;
import oshi.util.MapUtil;
import oshi.util.ParseUtil;
import oshi.util.platform.linux.ProcUtil;

public class LinuxOperatingSystem
extends AbstractOperatingSystem {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(LinuxOperatingSystem.class);
    protected String versionId;
    protected String codeName;
    private final int memoryPageSize;
    private static long hz = 1000L;
    private static long bootTime = 0L;
    private static final int[] PROC_PID_STAT_ORDERS = new int[ProcPidStat.values().length];
    private static int procPidStatLength;
    private transient LinuxUserGroupInfo userGroupInfo = new LinuxUserGroupInfo();

    public LinuxOperatingSystem() {
        this.manufacturer = "GNU/Linux";
        this.setFamilyFromReleaseFiles();
        this.version = new LinuxOSVersionInfoEx(this.versionId, this.codeName);
        this.memoryPageSize = LinuxOperatingSystem.getMemoryPageSize();
        LinuxOperatingSystem.init();
        this.initBitness();
    }

    private static void init() {
        File[] pids = ProcUtil.getPidFiles();
        Arrays.sort(pids, new Comparator<File>(){

            @Override
            public int compare(File f1, File f2) {
                return Integer.valueOf(f2.getName()).compareTo(Integer.valueOf(f1.getName()));
            }
        });
        long youngestJiffies = 0L;
        String youngestPid = "";
        for (File pid : pids) {
            int parenIndex;
            String procPidStat;
            String[] split;
            List<String> stat = FileUtil.readFile(String.format("/proc/%s/stat", pid.getName()), false);
            if (stat.isEmpty() || !stat.get(0).contains(")") || (split = ParseUtil.whitespaces.split((procPidStat = stat.get(0)).substring(parenIndex = procPidStat.lastIndexOf(41)))).length < 21) continue;
            youngestJiffies = ParseUtil.parseLongOrDefault(split[20], 0L);
            youngestPid = pid.getName();
            procPidStatLength = split.length + 1;
            break;
        }
        LOG.debug("Youngest PID is {} with {} jiffies", (Object)youngestPid, (Object)youngestJiffies);
        if (youngestJiffies == 0L) {
            LOG.error("Couldn't find any running processes, which is odd since we are in a running process. Process time values are in jiffies, not milliseconds.");
            return;
        }
        float startTimeSecsSinceBoot = ProcUtil.getSystemUptimeFromProc();
        bootTime = System.currentTimeMillis() - (long)(1000.0f * startTimeSecsSinceBoot);
        String etime = ExecutingCommand.getFirstAnswer(String.format("ps -p %s -o etimes=", youngestPid));
        if (!etime.isEmpty()) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Etime is {} seconds", (Object)etime.trim());
            }
            startTimeSecsSinceBoot -= Float.parseFloat(etime.trim());
        }
        LOG.debug("Start time in secs: {}", (Object)Float.valueOf(startTimeSecsSinceBoot));
        if (startTimeSecsSinceBoot <= 0.0f) {
            LOG.warn("Couldn't calculate jiffies per second. Process time values are in jiffies, not milliseconds.");
            return;
        }
        hz = (long)((float)youngestJiffies / startTimeSecsSinceBoot + 0.5f);
        if (hz == 0L) {
            hz = 1000L;
        }
    }

    private void initBitness() {
        if (this.bitness < 64 && ExecutingCommand.getFirstAnswer("uname -m").indexOf("64") != -1) {
            this.bitness = 64;
        }
    }

    private static int getMemoryPageSize() {
        try {
            return Libc.INSTANCE.getpagesize();
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            LOG.error("Failed to get the memory page size.", (Throwable)e);
            return 4096;
        }
    }

    @Override
    public FileSystem getFileSystem() {
        return new LinuxFileSystem();
    }

    @Override
    public OSProcess[] getProcesses(int limit, OperatingSystem.ProcessSort sort) {
        File[] pids;
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        for (File pid : pids = ProcUtil.getPidFiles()) {
            OSProcess proc = this.getProcess(ParseUtil.parseIntOrDefault(pid.getName(), 0));
            if (proc == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[sorted.size()]);
    }

    @Override
    public OSProcess getProcess(int pid) {
        String path = "";
        Memory buf = new Memory(1024L);
        int size = Libc.INSTANCE.readlink(String.format("/proc/%d/exe", pid), (Pointer)buf, 1023);
        if (size > 0) {
            path = buf.getString(0L).substring(0, size);
        }
        Map<String, String> io = FileUtil.getKeyValueMapFromFile(String.format("/proc/%d/io", pid), ":");
        long now = System.currentTimeMillis();
        String stat = FileUtil.getStringFromFile(String.format("/proc/%d/stat", pid));
        if (stat.isEmpty()) {
            return null;
        }
        OSProcess proc = new OSProcess();
        proc.setProcessID(pid);
        long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, procPidStatLength, ' ');
        proc.setParentProcessID((int)statArray[ProcPidStat.PPID.ordinal()]);
        proc.setThreadCount((int)statArray[ProcPidStat.THREAD_COUNT.ordinal()]);
        proc.setPriority((int)statArray[ProcPidStat.PRIORITY.ordinal()]);
        proc.setVirtualSize(statArray[ProcPidStat.VSZ.ordinal()]);
        proc.setResidentSetSize(statArray[ProcPidStat.RSS.ordinal()] * (long)this.memoryPageSize);
        proc.setKernelTime(statArray[ProcPidStat.KERNEL_TIME.ordinal()] * 1000L / hz);
        proc.setUserTime(statArray[ProcPidStat.USER_TIME.ordinal()] * 1000L / hz);
        proc.setStartTime(bootTime + statArray[ProcPidStat.START_TIME.ordinal()] * 1000L / hz);
        proc.setUpTime(now - proc.getStartTime());
        proc.setBytesRead(ParseUtil.parseLongOrDefault(MapUtil.getOrDefault(io, "read_bytes", ""), 0L));
        proc.setBytesWritten(ParseUtil.parseLongOrDefault(MapUtil.getOrDefault(io, "write_bytes", ""), 0L));
        List<String> openFilesList = ExecutingCommand.runNative(String.format("ls -f /proc/%d/fd", pid));
        proc.setOpenFiles((long)openFilesList.size() - 1L);
        Map<String, String> status = FileUtil.getKeyValueMapFromFile(String.format("/proc/%d/status", pid), ":");
        proc.setName(MapUtil.getOrDefault(status, "Name", ""));
        proc.setPath(path);
        switch (MapUtil.getOrDefault(status, "State", "U").charAt(0)) {
            case 'R': {
                proc.setState(OSProcess.State.RUNNING);
                break;
            }
            case 'S': {
                proc.setState(OSProcess.State.SLEEPING);
                break;
            }
            case 'D': {
                proc.setState(OSProcess.State.WAITING);
                break;
            }
            case 'Z': {
                proc.setState(OSProcess.State.ZOMBIE);
                break;
            }
            case 'T': {
                proc.setState(OSProcess.State.STOPPED);
                break;
            }
            default: {
                proc.setState(OSProcess.State.OTHER);
            }
        }
        proc.setUserID(ParseUtil.whitespaces.split(MapUtil.getOrDefault(status, "Uid", ""))[0]);
        proc.setGroupID(ParseUtil.whitespaces.split(MapUtil.getOrDefault(status, "Gid", ""))[0]);
        OSUser user = this.userGroupInfo.getUser(proc.getUserID());
        if (user != null) {
            proc.setUser(user.getUserName());
        }
        proc.setGroup(this.userGroupInfo.getGroupName(proc.getGroupID()));
        proc.setCommandLine(FileUtil.getStringFromFile(String.format("/proc/%d/cmdline", pid)));
        try {
            String cwdLink = String.format("/proc/%d/cwd", pid);
            String cwd = new File(cwdLink).getCanonicalPath();
            if (!cwd.equals(cwdLink)) {
                proc.setCurrentWorkingDirectory(cwd);
            }
        }
        catch (IOException e) {
            LOG.trace("Couldn't find cwd for pid {}: {}", (Object)pid, (Object)e);
        }
        return proc;
    }

    @Override
    public OSProcess[] getChildProcesses(int parentPid, int limit, OperatingSystem.ProcessSort sort) {
        File[] procFiles;
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        for (File procFile : procFiles = ProcUtil.getPidFiles()) {
            OSProcess proc;
            int pid = ParseUtil.parseIntOrDefault(procFile.getName(), 0);
            if (parentPid != LinuxOperatingSystem.getParentPidFromProcFile(pid) || (proc = this.getProcess(pid)) == null) continue;
            procs.add(proc);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[sorted.size()]);
    }

    private static int getParentPidFromProcFile(int pid) {
        String stat = FileUtil.getStringFromFile(String.format("/proc/%d/stat", pid));
        long[] statArray = ParseUtil.parseStringToLongArray(stat, PROC_PID_STAT_ORDERS, procPidStatLength, ' ');
        return (int)statArray[ProcPidStat.PPID.ordinal()];
    }

    @Override
    public int getProcessId() {
        return Libc.INSTANCE.getpid();
    }

    @Override
    public int getProcessCount() {
        return ProcUtil.getPidFiles().length;
    }

    @Override
    public int getThreadCount() {
        try {
            Libc.Sysinfo info = new Libc.Sysinfo();
            if (0 != Libc.INSTANCE.sysinfo(info)) {
                LOG.error("Failed to get process thread count. Error code: {}", (Object)Native.getLastError());
                return 0;
            }
            return info.procs;
        }
        catch (NoClassDefFoundError | UnsatisfiedLinkError e) {
            LOG.error("Failed to get procs from sysinfo. {}", (Throwable)e);
            return 0;
        }
    }

    @Override
    public NetworkParams getNetworkParams() {
        return new LinuxNetworkParams();
    }

    private void setFamilyFromReleaseFiles() {
        if (this.family == null) {
            if (this.readOsRelease()) {
                return;
            }
            if (this.execLsbRelease()) {
                return;
            }
            if (this.readLsbRelease()) {
                return;
            }
            String etcDistribRelease = LinuxOperatingSystem.getReleaseFilename();
            if (this.readDistribRelease(etcDistribRelease)) {
                return;
            }
            this.family = LinuxOperatingSystem.filenameToFamily(etcDistribRelease.replace("/etc/", "").replace("release", "").replace("version", "").replace("-", "").replace("_", ""));
        }
    }

    private boolean readOsRelease() {
        if (new File("/etc/os-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/os-release");
            for (String line : osRelease) {
                if (line.startsWith("VERSION=")) {
                    LOG.debug("os-release: {}", (Object)line);
                    line = line.replace("VERSION=", "").replaceAll("^\"|\"$", "").trim();
                    String[] split = line.split("[()]");
                    if (split.length <= 1) {
                        split = line.split(", ");
                    }
                    if (split.length > 0) {
                        this.versionId = split[0].trim();
                    }
                    if (split.length <= 1) continue;
                    this.codeName = split[1].trim();
                    continue;
                }
                if (line.startsWith("NAME=") && this.family == null) {
                    LOG.debug("os-release: {}", (Object)line);
                    this.family = line.replace("NAME=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("VERSION_ID=") || this.versionId != null) continue;
                LOG.debug("os-release: {}", (Object)line);
                this.versionId = line.replace("VERSION_ID=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean execLsbRelease() {
        for (String line : ExecutingCommand.runNative("lsb_release -a")) {
            if (line.startsWith("Description:")) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                if (!(line = line.replace("Description:", "").trim()).contains(" release ")) continue;
                this.family = this.parseRelease(line, " release ");
                continue;
            }
            if (line.startsWith("Distributor ID:") && this.family == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.family = line.replace("Distributor ID:", "").trim();
                continue;
            }
            if (line.startsWith("Release:") && this.versionId == null) {
                LOG.debug("lsb_release -a: {}", (Object)line);
                this.versionId = line.replace("Release:", "").trim();
                continue;
            }
            if (!line.startsWith("Codename:") || this.codeName != null) continue;
            LOG.debug("lsb_release -a: {}", (Object)line);
            this.codeName = line.replace("Codename:", "").trim();
        }
        return this.family != null;
    }

    private boolean readLsbRelease() {
        if (new File("/etc/lsb-release").exists()) {
            List<String> osRelease = FileUtil.readFile("/etc/lsb-release");
            for (String line : osRelease) {
                if (line.startsWith("DISTRIB_DESCRIPTION=")) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    if (!(line = line.replace("DISTRIB_DESCRIPTION=", "").replaceAll("^\"|\"$", "").trim()).contains(" release ")) continue;
                    this.family = this.parseRelease(line, " release ");
                    continue;
                }
                if (line.startsWith("DISTRIB_ID=") && this.family == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.family = line.replace("DISTRIB_ID=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (line.startsWith("DISTRIB_RELEASE=") && this.versionId == null) {
                    LOG.debug("lsb-release: {}", (Object)line);
                    this.versionId = line.replace("DISTRIB_RELEASE=", "").replaceAll("^\"|\"$", "").trim();
                    continue;
                }
                if (!line.startsWith("DISTRIB_CODENAME=") || this.codeName != null) continue;
                LOG.debug("lsb-release: {}", (Object)line);
                this.codeName = line.replace("DISTRIB_CODENAME=", "").replaceAll("^\"|\"$", "").trim();
            }
        }
        return this.family != null;
    }

    private boolean readDistribRelease(String filename) {
        if (new File(filename).exists()) {
            List<String> osRelease = FileUtil.readFile(filename);
            for (String line : osRelease) {
                LOG.debug("{}: {}", (Object)filename, (Object)line);
                if (line.contains(" release ")) {
                    this.family = this.parseRelease(line, " release ");
                    break;
                }
                if (!line.contains(" VERSION ")) continue;
                this.family = this.parseRelease(line, " VERSION ");
                break;
            }
        }
        return this.family != null;
    }

    private String parseRelease(String line, String splitLine) {
        String[] split = line.split(splitLine);
        String family = split[0].trim();
        if (split.length > 1) {
            if ((split = split[1].split("[()]")).length > 0) {
                this.versionId = split[0].trim();
            }
            if (split.length > 1) {
                this.codeName = split[1].trim();
            }
        }
        return family;
    }

    protected static String getReleaseFilename() {
        File etc = new File("/etc");
        File[] matchingFiles = etc.listFiles(new FileFilter(){

            @Override
            public boolean accept(File f) {
                return (f.getName().endsWith("-release") || f.getName().endsWith("-version") || f.getName().endsWith("_release") || f.getName().endsWith("_version")) && !f.getName().endsWith("os-release") && !f.getName().endsWith("lsb-release");
            }
        });
        if (matchingFiles != null && matchingFiles.length > 0) {
            return matchingFiles[0].getPath();
        }
        if (new File("/etc/release").exists()) {
            return "/etc/release";
        }
        return "/etc/issue";
    }

    private static String filenameToFamily(String name) {
        switch (name.toLowerCase()) {
            case "": {
                return "Solaris";
            }
            case "blackcat": {
                return "Black Cat";
            }
            case "bluewhite64": {
                return "BlueWhite64";
            }
            case "e-smith": {
                return "SME Server";
            }
            case "eos": {
                return "FreeEOS";
            }
            case "hlfs": {
                return "HLFS";
            }
            case "lfs": {
                return "Linux-From-Scratch";
            }
            case "linuxppc": {
                return "Linux-PPC";
            }
            case "meego": {
                return "MeeGo";
            }
            case "mandakelinux": {
                return "Mandrake";
            }
            case "mklinux": {
                return "MkLinux";
            }
            case "nld": {
                return "Novell Linux Desktop";
            }
            case "novell": 
            case "SuSE": {
                return "SUSE Linux";
            }
            case "pld": {
                return "PLD";
            }
            case "redhat": {
                return "Red Hat Linux";
            }
            case "sles": {
                return "SUSE Linux ES9";
            }
            case "sun": {
                return "Sun JDS";
            }
            case "synoinfo": {
                return "Synology";
            }
            case "tinysofa": {
                return "Tiny Sofa";
            }
            case "turbolinux": {
                return "TurboLinux";
            }
            case "ultrapenguin": {
                return "UltraPenguin";
            }
            case "va": {
                return "VA-Linux";
            }
            case "vmware": {
                return "VMWareESX";
            }
            case "yellowdog": {
                return "Yellow Dog";
            }
            case "issue": {
                return "Unknown";
            }
        }
        return name.substring(0, 1).toUpperCase() + name.substring(1);
    }

    public static long getHz() {
        return hz;
    }

    static {
        for (ProcPidStat stat : ProcPidStat.values()) {
            LinuxOperatingSystem.PROC_PID_STAT_ORDERS[stat.ordinal()] = stat.getOrder() - 1;
        }
        procPidStatLength = 52;
        LinuxOperatingSystem.init();
    }

    static enum ProcPidStat {
        PPID(4),
        USER_TIME(14),
        KERNEL_TIME(15),
        PRIORITY(18),
        THREAD_COUNT(20),
        START_TIME(22),
        VSZ(23),
        RSS(24);

        private int order;

        public int getOrder() {
            return this.order;
        }

        private ProcPidStat(int order) {
            this.order = order;
        }
    }
}

