/*
 * Decompiled with CFR 0.152.
 */
package oshi.software.os.unix.freebsd;

import com.sun.jna.Memory;
import com.sun.jna.Pointer;
import com.sun.jna.ptr.IntByReference;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import oshi.annotation.concurrent.ThreadSafe;
import oshi.driver.unix.freebsd.Who;
import oshi.jna.platform.unix.freebsd.FreeBsdLibc;
import oshi.software.common.AbstractOperatingSystem;
import oshi.software.os.FileSystem;
import oshi.software.os.InternetProtocolStats;
import oshi.software.os.NetworkParams;
import oshi.software.os.OSProcess;
import oshi.software.os.OSService;
import oshi.software.os.OSSession;
import oshi.software.os.OperatingSystem;
import oshi.software.os.unix.freebsd.FreeBsdFileSystem;
import oshi.software.os.unix.freebsd.FreeBsdInternetProtocolStats;
import oshi.software.os.unix.freebsd.FreeBsdNetworkParams;
import oshi.software.os.unix.freebsd.FreeBsdOSVersionInfoEx;
import oshi.util.ExecutingCommand;
import oshi.util.LsofUtil;
import oshi.util.ParseUtil;
import oshi.util.platform.unix.freebsd.BsdSysctlUtil;

@ThreadSafe
public class FreeBsdOperatingSystem
extends AbstractOperatingSystem {
    private static final Logger LOG = LoggerFactory.getLogger(FreeBsdOperatingSystem.class);
    private static final long BOOTTIME = FreeBsdOperatingSystem.querySystemBootTime();

    public FreeBsdOperatingSystem() {
        this.version = new FreeBsdOSVersionInfoEx();
    }

    @Override
    public String queryManufacturer() {
        return "Unix/BSD";
    }

    @Override
    public AbstractOperatingSystem.FamilyVersionInfo queryFamilyVersionInfo() {
        String family = BsdSysctlUtil.sysctl("kern.ostype", "FreeBSD");
        String version = BsdSysctlUtil.sysctl("kern.osrelease", "");
        String versionInfo = BsdSysctlUtil.sysctl("kern.version", "");
        String buildNumber = versionInfo.split(":")[0].replace(family, "").replace(version, "").trim();
        return new AbstractOperatingSystem.FamilyVersionInfo(family, new OperatingSystem.OSVersionInfo(version, null, buildNumber));
    }

    @Override
    protected int queryBitness(int jvmBitness) {
        if (jvmBitness < 64 && ExecutingCommand.getFirstAnswer("uname -m").indexOf("64") == -1) {
            return jvmBitness;
        }
        return 64;
    }

    @Override
    protected boolean queryElevated() {
        return System.getenv("SUDO_COMMAND") != null;
    }

    @Override
    public FileSystem getFileSystem() {
        return new FreeBsdFileSystem();
    }

    @Override
    public InternetProtocolStats getInternetProtocolStats() {
        return new FreeBsdInternetProtocolStats();
    }

    @Override
    public List<OSSession> getSessions() {
        return Collections.unmodifiableList(USE_WHO_COMMAND ? super.getSessions() : Who.queryUtxent());
    }

    @Override
    public OSProcess[] getProcesses(int limit, OperatingSystem.ProcessSort sort, boolean slowFields) {
        List<OSProcess> procs = this.getProcessListFromPS("ps -awwxo state,pid,ppid,user,uid,group,gid,nlwp,pri,vsz,rss,etimes,systime,time,comm,args", -1, slowFields);
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    @Override
    public OSProcess getProcess(int pid, boolean slowFields) {
        List<OSProcess> procs = this.getProcessListFromPS("ps -awwxo state,pid,ppid,user,uid,group,gid,nlwp,pri,vsz,rss,etimes,systime,time,comm,args -p ", pid, slowFields);
        if (procs.isEmpty()) {
            return null;
        }
        return procs.get(0);
    }

    @Override
    public OSProcess[] getChildProcesses(int parentPid, int limit, OperatingSystem.ProcessSort sort) {
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        for (OSProcess p : this.getProcesses(0, null)) {
            if (p.getParentProcessID() != parentPid) continue;
            procs.add(p);
        }
        List<OSProcess> sorted = this.processSort(procs, limit, sort);
        return sorted.toArray(new OSProcess[0]);
    }

    private List<OSProcess> getProcessListFromPS(String psCommand, int pid, boolean slowFields) {
        Map<Integer, String> cwdMap = LsofUtil.getCwdMap(pid);
        ArrayList<OSProcess> procs = new ArrayList<OSProcess>();
        List<String> procList = ExecutingCommand.runNative(psCommand + (pid < 0 ? "" : Integer.valueOf(pid)));
        if (procList.isEmpty() || procList.size() < 2) {
            return procs;
        }
        procList.remove(0);
        for (String proc : procList) {
            String[] split = ParseUtil.whitespaces.split(proc.trim(), 16);
            if (split.length < 16) continue;
            long now = System.currentTimeMillis();
            OSProcess fproc = new OSProcess(this);
            switch (split[0].charAt(0)) {
                case 'R': {
                    fproc.setState(OSProcess.State.RUNNING);
                    break;
                }
                case 'I': 
                case 'S': {
                    fproc.setState(OSProcess.State.SLEEPING);
                    break;
                }
                case 'D': 
                case 'L': 
                case 'U': {
                    fproc.setState(OSProcess.State.WAITING);
                    break;
                }
                case 'Z': {
                    fproc.setState(OSProcess.State.ZOMBIE);
                    break;
                }
                case 'T': {
                    fproc.setState(OSProcess.State.STOPPED);
                    break;
                }
                default: {
                    fproc.setState(OSProcess.State.OTHER);
                }
            }
            fproc.setProcessID(ParseUtil.parseIntOrDefault(split[1], 0));
            fproc.setParentProcessID(ParseUtil.parseIntOrDefault(split[2], 0));
            fproc.setUser(split[3]);
            fproc.setUserID(split[4]);
            fproc.setGroup(split[5]);
            fproc.setGroupID(split[6]);
            fproc.setThreadCount(ParseUtil.parseIntOrDefault(split[7], 0));
            fproc.setPriority(ParseUtil.parseIntOrDefault(split[8], 0));
            fproc.setVirtualSize(ParseUtil.parseLongOrDefault(split[9], 0L) * 1024L);
            fproc.setResidentSetSize(ParseUtil.parseLongOrDefault(split[10], 0L) * 1024L);
            long elapsedTime = ParseUtil.parseDHMSOrDefault(split[11], 0L);
            fproc.setUpTime(elapsedTime < 1L ? 1L : elapsedTime);
            fproc.setStartTime(now - fproc.getUpTime());
            fproc.setKernelTime(ParseUtil.parseDHMSOrDefault(split[12], 0L));
            fproc.setUserTime(ParseUtil.parseDHMSOrDefault(split[13], 0L) - fproc.getKernelTime());
            fproc.setPath(split[14]);
            fproc.setName(fproc.getPath().substring(fproc.getPath().lastIndexOf(47) + 1));
            fproc.setCommandLine(split[15]);
            fproc.setCurrentWorkingDirectory(cwdMap.getOrDefault(fproc.getProcessID(), ""));
            if (slowFields) {
                List<String> openFilesList = ExecutingCommand.runNative(String.format("lsof -p %d", pid));
                fproc.setOpenFiles((long)openFilesList.size() - 1L);
                int[] mib = new int[]{1, 14, 9, pid};
                Memory abi = new Memory(32L);
                IntByReference size = new IntByReference(32);
                if (0 == FreeBsdLibc.INSTANCE.sysctl(mib, mib.length, (Pointer)abi, size, null, 0)) {
                    String elf = abi.getString(0L);
                    if (elf.contains("ELF32")) {
                        fproc.setBitness(32);
                    } else if (elf.contains("ELF64")) {
                        fproc.setBitness(64);
                    }
                }
            }
            procs.add(fproc);
        }
        return procs;
    }

    @Override
    public long getProcessAffinityMask(int processId) {
        long bitMask = 0L;
        String cpuset = ExecutingCommand.getFirstAnswer("cpuset -gp " + processId);
        String[] split = cpuset.split(":");
        if (split.length > 1) {
            String[] bits;
            for (String bit : bits = split[1].split(",")) {
                int bitToSet = ParseUtil.parseIntOrDefault(bit.trim(), -1);
                if (bitToSet < 0) continue;
                bitMask |= 1L << bitToSet;
            }
        }
        return bitMask;
    }

    @Override
    public int getProcessId() {
        return FreeBsdLibc.INSTANCE.getpid();
    }

    @Override
    public int getProcessCount() {
        List<String> procList = ExecutingCommand.runNative("ps -axo pid");
        if (!procList.isEmpty()) {
            return procList.size() - 1;
        }
        return 0;
    }

    @Override
    public int getThreadCount() {
        int threads = 0;
        for (String proc : ExecutingCommand.runNative("ps -axo nlwp")) {
            threads += ParseUtil.parseIntOrDefault(proc.trim(), 0);
        }
        return threads;
    }

    @Override
    public long getSystemUptime() {
        return System.currentTimeMillis() / 1000L - BOOTTIME;
    }

    @Override
    public long getSystemBootTime() {
        return BOOTTIME;
    }

    private static long querySystemBootTime() {
        FreeBsdLibc.Timeval tv = new FreeBsdLibc.Timeval();
        if (!BsdSysctlUtil.sysctl("kern.boottime", tv) || tv.tv_sec == 0L) {
            return ParseUtil.parseLongOrDefault(ExecutingCommand.getFirstAnswer("sysctl -n kern.boottime").split(",")[0].replaceAll("\\D", ""), System.currentTimeMillis() / 1000L);
        }
        return tv.tv_sec;
    }

    @Override
    public NetworkParams getNetworkParams() {
        return new FreeBsdNetworkParams();
    }

    @Override
    public OSService[] getServices() {
        File[] listFiles;
        ArrayList<OSService> services = new ArrayList<OSService>();
        HashSet<String> running = new HashSet<String>();
        for (OSProcess p : this.getChildProcesses(1, 0, OperatingSystem.ProcessSort.PID)) {
            OSService s = new OSService(p.getName(), p.getProcessID(), OSService.State.RUNNING);
            services.add(s);
            running.add(p.getName());
        }
        File dir = new File("/etc/rc.d");
        if (dir.exists() && dir.isDirectory() && (listFiles = dir.listFiles()) != null) {
            for (File f : listFiles) {
                String name = f.getName();
                if (running.contains(name)) continue;
                OSService s = new OSService(name, 0, OSService.State.STOPPED);
                services.add(s);
            }
        } else {
            LOG.error("Directory: /etc/init does not exist");
        }
        return services.toArray(new OSService[0]);
    }
}

