/*
 * Decompiled with CFR 0.152.
 */
package org.openpdf.util;

import com.google.errorprone.annotations.CheckReturnValue;
import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.GraphicsDevice;
import java.awt.GraphicsEnvironment;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Map;
import java.util.function.Consumer;
import java.util.logging.Level;
import javax.imageio.ImageIO;
import org.jspecify.annotations.Nullable;
import org.openpdf.util.Configuration;
import org.openpdf.util.DownscaleQuality;
import org.openpdf.util.ScalingOptions;
import org.openpdf.util.XRLog;

public class ImageUtil {
    private static final Map<DownscaleQuality, Scaler> qualities = Map.of(DownscaleQuality.FAST, new OldScaler(), DownscaleQuality.HIGH_QUALITY, new HighQualityScaler(), DownscaleQuality.LOW_QUALITY, new FastScaler(), DownscaleQuality.AREA, new AreaAverageScaler());

    public static void clearImage(BufferedImage image) {
        ImageUtil.withGraphics(image, (Graphics2D g2d) -> {
            g2d.setColor(Color.WHITE);
            g2d.fillRect(0, 0, image.getWidth(), image.getHeight());
        });
    }

    public static void withGraphics(BufferedImage image, Consumer<Graphics2D> block) {
        Graphics2D g = (Graphics2D)image.getGraphics();
        try {
            block.accept(g);
        }
        finally {
            g.dispose();
        }
    }

    public static void withGraphics(Graphics graphics, Consumer<Graphics> block) {
        Graphics g = graphics.create();
        try {
            block.accept(g);
        }
        finally {
            g.dispose();
        }
    }

    @CheckReturnValue
    public static BufferedImage makeCompatible(BufferedImage image) {
        BufferedImage cimg;
        if (GraphicsEnvironment.isHeadless()) {
            cimg = ImageUtil.createCompatibleBufferedImage(image.getWidth(), image.getHeight(), image.getTransparency());
        } else {
            GraphicsConfiguration gc = ImageUtil.getGraphicsConfiguration();
            if (image.getColorModel().equals(gc.getColorModel())) {
                return image;
            }
            cimg = gc.createCompatibleImage(image.getWidth(), image.getHeight(), image.getTransparency());
        }
        ImageUtil.withGraphics(cimg, (Graphics2D cg) -> cg.drawImage(image, 0, 0, null));
        return cimg;
    }

    @CheckReturnValue
    public static BufferedImage createCompatibleBufferedImage(int width, int height, int biType) {
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        if (ge.isHeadlessInstance()) {
            return new BufferedImage(width, height, biType);
        }
        GraphicsConfiguration gc = ImageUtil.getGraphicsConfiguration();
        return gc.createCompatibleImage(width, height, ImageUtil.detectTransparency(biType));
    }

    static int detectTransparency(int biType) {
        return switch (biType) {
            case 2, 3, 6, 7 -> 3;
            default -> 1;
        };
    }

    @CheckReturnValue
    private static GraphicsConfiguration getGraphicsConfiguration() {
        GraphicsEnvironment ge = GraphicsEnvironment.getLocalGraphicsEnvironment();
        GraphicsDevice gs = ge.getDefaultScreenDevice();
        return gs.getDefaultConfiguration();
    }

    @CheckReturnValue
    public static BufferedImage createCompatibleBufferedImage(int width, int height) {
        return ImageUtil.createCompatibleBufferedImage(width, height, 2);
    }

    @CheckReturnValue
    public static BufferedImage getScaledInstance(ScalingOptions opt, BufferedImage orgImage) {
        int h;
        int w = orgImage.getWidth(null);
        if (opt.sizeMatches(w, h = orgImage.getHeight(null))) {
            return orgImage;
        }
        ScalingOptions normalizedOptions = opt.withTarget(opt.getTargetWidth() <= 0 ? w : opt.getTargetWidth(), opt.getTargetHeight() <= 0 ? h : opt.getTargetHeight());
        Scaler scaler = qualities.get(opt.getDownscalingHint());
        return scaler.getScaledInstance(orgImage, normalizedOptions);
    }

    @CheckReturnValue
    public static BufferedImage getScaledInstance(BufferedImage orgImage, int targetWidth, int targetHeight) {
        String downscaleQuality = Configuration.valueFor("xr.image.scale", DownscaleQuality.HIGH_QUALITY.asString());
        DownscaleQuality quality = DownscaleQuality.forString(downscaleQuality, DownscaleQuality.HIGH_QUALITY);
        Object hint = Configuration.valueFromClassConstant("xr.image.render-quality", RenderingHints.VALUE_INTERPOLATION_BICUBIC);
        ScalingOptions opt = new ScalingOptions(targetWidth, targetHeight, quality, hint);
        return ImageUtil.getScaledInstance(opt, orgImage);
    }

    @CheckReturnValue
    public static BufferedImage convertToBufferedImage(Image awtImg, int type) {
        if (awtImg instanceof BufferedImage) {
            BufferedImage result = (BufferedImage)awtImg;
            return result;
        }
        BufferedImage image = ImageUtil.createCompatibleBufferedImage(awtImg.getWidth(null), awtImg.getHeight(null), type);
        ImageUtil.withGraphics(image, (Graphics2D g) -> g.drawImage(awtImg, 0, 0, null, null));
        return image;
    }

    @CheckReturnValue
    public static BufferedImage createTransparentImage(int width, int height) {
        BufferedImage bi = ImageUtil.createCompatibleBufferedImage(width, height, 2);
        ImageUtil.withGraphics(bi, (Graphics2D g2d) -> {
            Color transparent = new Color(0, 0, 0, 0);
            g2d.setColor(transparent);
            g2d.setComposite(AlphaComposite.Src);
            g2d.fillRect(0, 0, width, height);
        });
        return bi;
    }

    @CheckReturnValue
    public static boolean isEmbeddedBase64Image(@Nullable String uri) {
        return uri != null && uri.startsWith("data:image");
    }

    @CheckReturnValue
    public static byte @Nullable [] getEmbeddedBase64Image(String imageDataUri) {
        int b64Index = imageDataUri.indexOf("base64,");
        if (b64Index != -1) {
            String b64encoded = imageDataUri.substring(b64Index + "base64,".length());
            if (b64encoded.contains("%")) {
                b64encoded = URLDecoder.decode(b64encoded, StandardCharsets.US_ASCII);
            }
            return Base64.getDecoder().decode(b64encoded);
        }
        XRLog.load(Level.SEVERE, "Embedded XHTML images must be encoded in base 64.");
        return null;
    }

    @CheckReturnValue
    public static @Nullable BufferedImage loadEmbeddedBase64Image(String imageDataUri) {
        try {
            byte[] buffer = ImageUtil.getEmbeddedBase64Image(imageDataUri);
            if (buffer != null) {
                return ImageIO.read(new ByteArrayInputStream(buffer));
            }
        }
        catch (IOException ex) {
            XRLog.exception("Can't read XHTML embedded image", ex);
        }
        return null;
    }

    private static interface Scaler {
        @CheckReturnValue
        public BufferedImage getScaledInstance(BufferedImage var1, ScalingOptions var2);
    }

    private static class OldScaler
    extends AbstractFastScaler {
        private OldScaler() {
        }

        @Override
        @CheckReturnValue
        protected int getImageScalingMethod() {
            return 2;
        }
    }

    private static class HighQualityScaler
    implements Scaler {
        private HighQualityScaler() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        @CheckReturnValue
        public BufferedImage getScaledInstance(BufferedImage img, ScalingOptions opt) {
            int h;
            int w;
            int width = img.getWidth(null);
            int height = img.getHeight(null);
            if (opt.getTargetWidth() < width && opt.getTargetHeight() < height) {
                w = width;
                h = height;
            } else {
                w = opt.getTargetWidth();
                h = opt.getTargetHeight();
            }
            BufferedImage scaled = img;
            do {
                if (w > opt.getTargetWidth() && (w /= 2) < opt.getTargetWidth()) {
                    w = opt.getTargetWidth();
                }
                if (h > opt.getTargetHeight() && (h /= 2) < opt.getTargetHeight()) {
                    h = opt.getTargetHeight();
                }
                BufferedImage tmp = ImageUtil.createCompatibleBufferedImage(w, h, img.getType());
                Graphics2D g2 = tmp.createGraphics();
                try {
                    opt.applyRenderingHints(g2);
                    g2.drawImage(scaled, 0, 0, w, h, null);
                }
                finally {
                    g2.dispose();
                }
                scaled = tmp;
            } while (w != opt.getTargetWidth() || h != opt.getTargetHeight());
            return scaled;
        }
    }

    private static class FastScaler
    implements Scaler {
        private FastScaler() {
        }

        @Override
        @CheckReturnValue
        public BufferedImage getScaledInstance(BufferedImage img, ScalingOptions opt) {
            int w = opt.getTargetWidth();
            int h = opt.getTargetHeight();
            BufferedImage scaled = ImageUtil.createCompatibleBufferedImage(w, h, img.getType());
            ImageUtil.withGraphics(scaled, g2 -> {
                opt.applyRenderingHints((Graphics2D)g2);
                g2.drawImage(img, 0, 0, w, h, null);
            });
            return scaled;
        }
    }

    private static class AreaAverageScaler
    extends AbstractFastScaler {
        private AreaAverageScaler() {
        }

        @Override
        @CheckReturnValue
        protected int getImageScalingMethod() {
            return 16;
        }
    }

    private static abstract class AbstractFastScaler
    implements Scaler {
        private AbstractFastScaler() {
        }

        @Override
        @CheckReturnValue
        public BufferedImage getScaledInstance(BufferedImage img, ScalingOptions opt) {
            Image scaled = img.getScaledInstance(opt.getTargetWidth(), opt.getTargetHeight(), this.getImageScalingMethod());
            return ImageUtil.convertToBufferedImage(scaled, img.getType());
        }

        protected abstract int getImageScalingMethod();
    }
}

