package in.kyle.api.bukkit.plugin;

import org.bukkit.event.Cancellable;
import org.bukkit.event.Event;
import org.bukkit.event.EventHandler;
import org.bukkit.event.EventPriority;
import org.bukkit.event.Listener;
import org.bukkit.plugin.EventExecutor;
import org.bukkit.plugin.Plugin;
import org.bukkit.plugin.PluginManager;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;

import in.kyle.api.utils.Try;
import lombok.AllArgsConstructor;
import lombok.Data;

@AllArgsConstructor
@Data
public abstract class TestPluginManager implements PluginManager {
    
    private final Map<Listener, Set<Consumer<Event>>> listeners = new HashMap<>();
    private final Set<Plugin> plugins = new HashSet<>();
    
    @Override
    public Plugin getPlugin(String name) {
        return plugins.stream()
                      .filter(plugin -> plugin.getName().equals(name))
                      .findAny()
                      .orElse(null);
    }
    
    @Override
    public Plugin[] getPlugins() {
        return plugins.toArray(new Plugin[plugins.size()]);
    }
    
    @Override
    public boolean isPluginEnabled(String name) {
        return getPlugin(name) != null;
    }
    
    @Override
    public boolean isPluginEnabled(Plugin plugin) {
        return plugins.contains(plugin);
    }
    
    @Override
    public void disablePlugins() {
        for (Plugin plugin : plugins) {
            plugin.onDisable();
        }
    }
    
    @Override
    public void clearPlugins() {
        disablePlugins();
        plugins.clear();
    }
    
    @Override
    public void callEvent(Event event) throws IllegalStateException {
        Set<Consumer<Event>> run = new HashSet<>();
        listeners.values().forEach(run::addAll);
        run.forEach(c -> c.accept(event));
    }
    
    
    @Override
    public void registerEvents(Listener listener, Plugin plugin) {
        for (Method method : listener.getClass().getDeclaredMethods()) {
            if (method.isAnnotationPresent(EventHandler.class) && method.getParameterCount() == 1) {
                EventHandler annotation = method.getAnnotation(EventHandler.class);
                Class<? extends Event> clazz =
                        (Class<? extends Event>) method.getParameterTypes()[0];
                registerEvent(clazz,
                              listener,
                              annotation.priority(),
                              (listener1, event) -> Try.to(() -> method.invoke(listener, event)),
                              plugin,
                              annotation.ignoreCancelled());
            }
        }
    }
    
    @Override
    public void registerEvent(Class<? extends Event> event,
                              Listener listener,
                              EventPriority priority,
                              EventExecutor executor,
                              Plugin plugin) {
        registerEvent(event, listener, priority, executor, plugin, false);
    }
    
    @Override
    public void registerEvent(Class<? extends Event> event,
                              Listener listener,
                              EventPriority priority,
                              EventExecutor executor,
                              Plugin plugin,
                              boolean ignoreCancelled) {
        Set<Consumer<Event>> consumers = listeners.getOrDefault(listener, new HashSet<>());
        consumers.add(e -> {
            if (e.getClass().equals(event)) {
                if (e instanceof Cancellable && !ignoreCancelled &&
                    ((Cancellable) e).isCancelled()) {
                    return;
                }
                Try.to(() -> executor.execute(listener, e));
            }
        });
        listeners.put(listener, consumers);
    }
    
    @Override
    public void enablePlugin(Plugin plugin) {
        if (!isPluginEnabled(plugin)) {
            plugins.add(plugin);
            plugin.onEnable();
        } else {
            throw new IllegalStateException("Plugin already enabled");
        }
    }
    
    @Override
    public void disablePlugin(Plugin plugin) {
        if (isPluginEnabled(plugin)) {
            plugins.add(plugin);
            plugin.onEnable();
        } else {
            throw new IllegalStateException("Plugin already enabled");
        }
    }
    
    @Override
    public boolean useTimings() {
        return false;
    }
}
