package in.kyle.api.bukkit.inventory;

import org.bukkit.Location;
import org.bukkit.Material;
import org.bukkit.entity.HumanEntity;
import org.bukkit.event.inventory.InventoryType;
import org.bukkit.inventory.InventoryHolder;
import org.bukkit.inventory.ItemStack;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.ListIterator;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

import in.kyle.api.bukkit.util.BoundedArrayList;
import in.kyle.api.generate.NotImplementedException;
import lombok.Data;
import lombok.RequiredArgsConstructor;

@Data
@RequiredArgsConstructor
public class CommonInventory {
    
    private static final AtomicInteger idIndex = new AtomicInteger();
    
    private final InventoryType type;
    private final InventoryHolder holder;
    private final List<ItemStack> items;
    
    private final List<HumanEntity> viewers = new ArrayList<>();
    private final String name = "inv-" + idIndex.incrementAndGet();
    
    private final String title;
    private final int size;
    
    private int maxStackSize;
    private Location location;
    
    public CommonInventory(InventoryType type, InventoryHolder holder, String title) {
        this(type,
             holder,
             new BoundedArrayList<>(type.getDefaultSize()),
             title,
             type.getDefaultSize());
    }
    
    public CommonInventory(InventoryType type, InventoryHolder holder) {
        this(type, holder, type.getDefaultTitle());
    }
    
    
    public ItemStack getItem(int index) {
        return items.get(index);
    }
    
    public void setItem(int index, ItemStack item) {
        items.set(index, item);
    }
    
    
    public HashMap<Integer, ItemStack> addItem(ItemStack... items) throws IllegalArgumentException {
        throw new NotImplementedException("Method not implemented");
    }
    
    
    public HashMap<Integer, ItemStack> removeItem(ItemStack... items)
            throws IllegalArgumentException {
        throw new NotImplementedException("Method not implemented");
    }
    
    
    public ItemStack[] getContents() {
        return items.toArray(new ItemStack[items.size()]);
    }
    
    
    public void setContents(ItemStack[] items) throws IllegalArgumentException {
        for (int i = 0; i < items.length; i++) {
            this.items.set(i, items[i]);
        }
    }
    
    
    public ItemStack[] getStorageContents() {
        return getContents();
    }
    
    
    public void setStorageContents(ItemStack[] items) throws IllegalArgumentException {
        setContents(items);
    }
    
    
    public boolean contains(int materialId) {
        return items.stream().anyMatch(itemStack -> itemStack.getType().getId() == materialId);
    }
    
    
    public boolean contains(Material material) throws IllegalArgumentException {
        return contains(material.getId());
    }
    
    
    public boolean contains(ItemStack item) {
        return items.stream().anyMatch(itemStack -> itemStack.equals(item));
    }
    
    private int getAmount(ItemStack is) {
        return (int) items.stream().filter(itemStack -> itemStack.equals(is)).count();
    }
    
    private int getAmount(Material material) {
        return (int) items.stream().filter(itemStack -> itemStack.getType() == material).count();
    }
    
    
    public boolean contains(int materialId, int amount) {
        return getAmount(Material.getMaterial(materialId)) >= amount;
    }
    
    
    public boolean contains(Material material, int amount) throws IllegalArgumentException {
        return contains(material.getId(), amount);
    }
    
    
    public boolean contains(ItemStack item, int amount) {
        return getAmount(item) >= amount;
    }
    
    
    public boolean containsAtLeast(ItemStack item, int amount) {
        return contains(item, amount);
    }
    
    
    public HashMap<Integer, ? extends ItemStack> all(int materialId) {
        return new HashMap<>(items.stream()
                                  .filter(item -> item.getTypeId() == materialId)
                                  .collect(Collectors.toMap(items::indexOf, item -> item)));
    }
    
    
    public HashMap<Integer, ? extends ItemStack> all(Material material)
            throws IllegalArgumentException {
        return all(material.getId());
    }
    
    
    public HashMap<Integer, ? extends ItemStack> all(ItemStack itemStack) {
        return new HashMap<>(items.stream()
                                  .filter(item -> item.equals(itemStack))
                                  .collect(Collectors.toMap(items::indexOf, item -> item)));
    }
    
    
    public int first(int materialId) {
        return items.stream()
                    .filter(item -> item.getTypeId() == materialId)
                    .findFirst()
                    .map(items::indexOf)
                    .orElse(-1);
    }
    
    
    public int first(Material material) throws IllegalArgumentException {
        return first(material.getId());
    }
    
    
    public int first(ItemStack item) {
        return items.stream()
                    .filter(i -> i.equals(item))
                    .findFirst()
                    .map(items::indexOf)
                    .orElse(-1);
    }
    
    
    public int firstEmpty() {
        return items.stream()
                    .filter(i -> i.getTypeId() == 0)
                    .findFirst()
                    .map(items::indexOf)
                    .orElse(-1);
    }
    
    
    public void remove(int materialId) {
        items.removeIf(itemStack -> itemStack.getTypeId() == materialId);
    }
    
    
    public void remove(Material material) throws IllegalArgumentException {
        items.removeIf(itemStack -> itemStack.getType() == material);
    }
    
    
    public void remove(ItemStack item) {
        items.removeIf(itemStack -> itemStack.equals(item));
    }
    
    
    public void clear(int index) {
        setItem(index, new ItemStack(Material.AIR));
    }
    
    
    public void clear() {
        for (int i = 0; i < items.size(); i++) {
            clear(i);
        }
    }
    
    
    public ListIterator<ItemStack> iterator() {
        return items.listIterator();
    }
    
    
    public ListIterator<ItemStack> iterator(int index) {
        return items.listIterator(index);
    }
}
