package org.komputing.kethereum.erc181.registrar

import kotlin.Boolean
import kotlin.ByteArray
import kotlin.String
import org.kethereum.contract.abi.types.PaginatedByteArray
import org.kethereum.contract.abi.types.encodeTypes
import org.kethereum.contract.abi.types.model.type_params.BitsTypeParams
import org.kethereum.contract.abi.types.model.type_params.BytesTypeParams
import org.kethereum.contract.abi.types.model.types.AddressETHType
import org.kethereum.contract.abi.types.model.types.BytesETHType
import org.kethereum.contract.abi.types.model.types.StringETHType
import org.kethereum.model.Address
import org.kethereum.model.Transaction
import org.kethereum.model.createEmptyTransaction
import org.kethereum.rpc.EthereumRPC

val FourByteClaimWithResolver: ByteArray = byteArrayOf(15, 90, 84, 102)

val FourByteClaim: ByteArray = byteArrayOf(30, -125, 64, -102)

val FourByteEns: ByteArray = byteArrayOf(63, 21, 69, 127)

val FourByteDefaultResolver: ByteArray = byteArrayOf(-126, -114, -85, 14)

val FourByteNode: ByteArray = byteArrayOf(-65, -5, -26, 28)

val FourByteSetName: ByteArray = byteArrayOf(-60, 127, 0, 39)

class ENSReverseRegistrarTransactionGenerator(
  address: Address
) {
  private val tx: Transaction = createEmptyTransaction().apply { to = address }

  internal fun claimWithResolverETHTyped(owner: AddressETHType, resolver: AddressETHType) =
      tx.copy(input = FourByteClaimWithResolver + encodeTypes(owner, resolver))
  /**
   * Signature: claimWithResolver(address,address)
   * 4Byte: 0f5a5466
   */
  fun claimWithResolver(owner: Address, resolver: Address): Transaction =
      claimWithResolverETHTyped(AddressETHType.ofNativeKotlinType(owner),
      AddressETHType.ofNativeKotlinType(resolver))

  internal fun claimETHTyped(owner: AddressETHType) = tx.copy(input = FourByteClaim +
      encodeTypes(owner))
  /**
   * Signature: claim(address)
   * 4Byte: 1e83409a
   */
  fun claim(owner: Address): Transaction = claimETHTyped(AddressETHType.ofNativeKotlinType(owner))

  internal fun ensETHTyped() = tx.copy(input = FourByteEns + encodeTypes())
  /**
   * Signature: ens()
   * 4Byte: 3f15457f
   */
  fun ens(): Transaction = ensETHTyped()

  internal fun defaultResolverETHTyped() = tx.copy(input = FourByteDefaultResolver + encodeTypes())
  /**
   * Signature: defaultResolver()
   * 4Byte: 828eab0e
   */
  fun defaultResolver(): Transaction = defaultResolverETHTyped()

  internal fun nodeETHTyped(addr: AddressETHType) = tx.copy(input = FourByteNode +
      encodeTypes(addr))
  /**
   * Signature: node(address)
   * 4Byte: bffbe61c
   */
  fun node(addr: Address): Transaction = nodeETHTyped(AddressETHType.ofNativeKotlinType(addr))

  internal fun setNameETHTyped(name: StringETHType) = tx.copy(input = FourByteSetName +
      encodeTypes(name))
  /**
   * Signature: setName(string)
   * 4Byte: c47f0027
   */
  fun setName(name: String): Transaction = setNameETHTyped(StringETHType.ofNativeKotlinType(name))
}

class ENSReverseRegistrarRPCConnector(
  private val address: Address,
  private val rpc: EthereumRPC
) {
  private val txGenerator: ENSReverseRegistrarTransactionGenerator =
      ENSReverseRegistrarTransactionGenerator(address)

  private fun claimWithResolverETHTyped(
    owner: AddressETHType,
    resolver: AddressETHType,
    blockSpec: String = "latest"
  ): BytesETHType? {
    val tx = txGenerator.claimWithResolverETHTyped(owner, resolver)
    return BytesETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BytesTypeParams(bytes=32))
  }

  /**
   * Signature: claimWithResolver(address,address)
   * 4Byte: 0f5a5466
   */
  fun claimWithResolver(
    owner: Address,
    resolver: Address,
    blockSpec: String = "latest"
  ): ByteArray? = claimWithResolverETHTyped(AddressETHType.ofNativeKotlinType(owner),
      AddressETHType.ofNativeKotlinType(resolver),blockSpec)?.toKotlinType()

  private fun claimETHTyped(owner: AddressETHType, blockSpec: String = "latest"): BytesETHType? {
    val tx = txGenerator.claimETHTyped(owner)
    return BytesETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BytesTypeParams(bytes=32))
  }

  /**
   * Signature: claim(address)
   * 4Byte: 1e83409a
   */
  fun claim(owner: Address, blockSpec: String = "latest"): ByteArray? =
      claimETHTyped(AddressETHType.ofNativeKotlinType(owner),blockSpec)?.toKotlinType()

  private fun ensETHTyped(blockSpec: String = "latest"): AddressETHType? {
    val tx = txGenerator.ensETHTyped()
    return AddressETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: ens()
   * 4Byte: 3f15457f
   */
  fun ens(blockSpec: String = "latest"): Address? = ensETHTyped(blockSpec)?.toKotlinType()

  private fun defaultResolverETHTyped(blockSpec: String = "latest"): AddressETHType? {
    val tx = txGenerator.defaultResolverETHTyped()
    return AddressETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx, blockSpec)))
  }

  /**
   * Signature: defaultResolver()
   * 4Byte: 828eab0e
   */
  fun defaultResolver(blockSpec: String = "latest"): Address? =
      defaultResolverETHTyped(blockSpec)?.toKotlinType()

  private fun nodeETHTyped(addr: AddressETHType, blockSpec: String = "latest"): BytesETHType? {
    val tx = txGenerator.nodeETHTyped(addr)
    return BytesETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BytesTypeParams(bytes=32))
  }

  /**
   * Signature: node(address)
   * 4Byte: bffbe61c
   */
  fun node(addr: Address, blockSpec: String = "latest"): ByteArray? =
      nodeETHTyped(AddressETHType.ofNativeKotlinType(addr),blockSpec)?.toKotlinType()

  private fun setNameETHTyped(name: StringETHType, blockSpec: String = "latest"): BytesETHType? {
    val tx = txGenerator.setNameETHTyped(name)
    return BytesETHType.ofPaginatedByteArray(PaginatedByteArray(rpc.call(tx,
        blockSpec)),BytesTypeParams(bytes=32))
  }

  /**
   * Signature: setName(string)
   * 4Byte: c47f0027
   */
  fun setName(name: String, blockSpec: String = "latest"): ByteArray? =
      setNameETHTyped(StringETHType.ofNativeKotlinType(name),blockSpec)?.toKotlinType()
}

class ENSReverseRegistrarTransactionDecoder {
  fun isClaimWithResolver(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteClaimWithResolver)

  fun isClaim(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteClaim)

  fun isEns(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteEns)

  fun isDefaultResolver(tx: Transaction): Boolean =
      tx.input.sliceArray(0..3).contentEquals(FourByteDefaultResolver)

  fun isNode(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteNode)

  fun isSetName(tx: Transaction): Boolean = tx.input.sliceArray(0..3).contentEquals(FourByteSetName)
}
