package com.github.kalaganov.circular

import android.content.Context
import android.content.res.TypedArray
import android.graphics.*
import android.graphics.drawable.Drawable
import android.support.v4.content.ContextCompat
import android.support.v7.widget.AppCompatTextView
import android.util.AttributeSet
import android.view.Gravity

/**
 * Created by Eugene on 10.05.2018
 */

class ProgressView @JvmOverloads constructor(context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0)
    : AppCompatTextView(context, attrs, defStyleAttr) {

    private val defaultSize = resources.getDimensionPixelSize(R.dimen.progress_view_default_size)
    private val backgroundPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply { style = Paint.Style.STROKE }
    private val progressPaint = Paint(Paint.ANTI_ALIAS_FLAG).apply { style = Paint.Style.STROKE }
    private val progressOnTipDrawablePaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG) }
    private val progressOnTipPaint by lazy { Paint(Paint.ANTI_ALIAS_FLAG).apply { textAlign = Paint.Align.CENTER } }
    private val progressOnTipBounds by lazy { Rect() }
    private val progressRectF = RectF()
    private var progressOnTipBitmap: Bitmap? = null
    private var progressFormatter: ProgressFormatter = DefaultProgressFormatter
    private var progressOnTipFormatter: ProgressFormatter = DefaultProgressFormatter

    var progressBackgroundStyle: BackgroundStyle = BackgroundStyle.STROKE
        set(value) {
            field = value
            backgroundPaint.style = when (value.ordinal) {
                1 -> Paint.Style.FILL
                2 -> Paint.Style.FILL_AND_STROKE
                else -> Paint.Style.STROKE
            }
            invalidate()
        }

    var progressStyle: ProgressStyle = ProgressStyle.STROKE
        set(value) {
            field = value
            progressPaint.style = when (value.ordinal) {
                1 -> Paint.Style.FILL
                2 -> Paint.Style.FILL_AND_STROKE
                else -> Paint.Style.STROKE
            }
            invalidate()
        }

    var progressBackgroundColor: Int
        set(value) {
            backgroundPaint.color = value
            invalidate()
        }
        get() = backgroundPaint.color

    var progressBackgroundStrokeWidth: Float
        set(value) {
            backgroundPaint.strokeWidth = value
            invalidate()
        }
        get() = backgroundPaint.strokeWidth

    var progressColor: Int
        set(value) {
            progressPaint.color = value
            invalidate()
        }
        get() = progressPaint.color


    var progressStrokeWidth: Float
        set(value) {
            progressPaint.strokeWidth = value
            invalidate()
        }
        get() = progressPaint.strokeWidth


    var progressStartAngle = 0f
        set(value) {
            field = value - 90
            invalidate()
        }

    var minProgressValue = 0f
        set(value) {
            field = value
            invalidate()
        }

    var maxProgressValue = 0f
        set(value) {
            field = value
            invalidate()
        }

    var progress = 0f
        set(value) {
            field = value
            invalidate()
        }

    var progressCounterClockwise = false
        set(value) {
            field = value
            invalidate()
        }

    var progressStrokeCapRound = false
        set(value) {
            field = value
            invalidate()
        }

    var showProgressOnTip = false
        set(value) {
            field = value
            invalidate()
        }

    var progressOnTipTextColor
        set(value) {
            progressOnTipPaint.color = value
            invalidate()
        }
        get() = progressOnTipPaint.color


    var progressOnTipTextSize
        set(value) {
            progressOnTipPaint.textSize = value
            invalidate()
        }
        get() = progressOnTipPaint.textSize

    var progressOnTipText: String? = ""
        set(value) {
            field = value
            invalidate()
        }
        get() = if (field.isNullOrEmpty()) progressOnTipFormatter.format(progress, minProgressValue, maxProgressValue) else field

    var progressOnTipDrawable: Drawable? = null
        set(value) {
            field = value
            progressOnTipBitmap = if (value == null) null else drawableToBitmap(value)
            invalidate()
        }

    var progressOnTipDrawableRes: Int = 0
        set(value) {
            field = value
            progressOnTipDrawable = ContextCompat.getDrawable(context, value)
        }

    fun setProgressFormatter(formatter: ProgressFormatter?) {
        progressFormatter = formatter ?: DefaultProgressFormatter
        invalidate()
    }

    fun setProgressOnTipFormatter(formatter: ProgressFormatter?) {
        progressOnTipFormatter = formatter ?: DefaultProgressFormatter
        invalidate()
    }

    var showProgressAsPercentage: Boolean = false
        set(value) {
            setProgressFormatter(if (value) PercentProgressFormatter else DefaultProgressFormatter)
        }

    var showProgressOnTipAsPercentage: Boolean = false
        set(value) {
            setProgressOnTipFormatter(if (value) PercentProgressFormatter else DefaultProgressFormatter)
        }

    init {
        val typedArray = context.theme.obtainStyledAttributes(attrs, R.styleable.ProgressView, 0, R.style.ProgressViewDefaultStyle)
        typedArray?.let { arr -> applyAttrs(arr) }
        includeFontPadding = false
        gravity = Gravity.CENTER
    }

    private fun applyAttrs(typedArray: TypedArray) {
        progress = typedArray.getFloat(R.styleable.ProgressView_progress, 0f)
        minProgressValue = typedArray.getFloat(R.styleable.ProgressView_minProgressValue, 0f)
        maxProgressValue = typedArray.getFloat(R.styleable.ProgressView_maxProgressValue, 0f)

        val backProgressStyle = typedArray.getInt(R.styleable.ProgressView_progressBackgroundStyle, 0)
        progressBackgroundStyle = BackgroundStyle.values()[backProgressStyle]

        backgroundPaint.strokeWidth = typedArray.getDimension(R.styleable.ProgressView_progressBackgroundStrokeWidth, 0f)
        backgroundPaint.color = typedArray.getColor(R.styleable.ProgressView_progressBackgroundColor, Color.BLACK)

        val style = typedArray.getInt(R.styleable.ProgressView_progressStyle, 0)
        progressStyle = ProgressStyle.values()[style]

        progressPaint.strokeWidth = typedArray.getDimension(R.styleable.ProgressView_progressStrokeWidth, 0f)
        progressPaint.color = typedArray.getColor(R.styleable.ProgressView_progressColor, Color.BLACK)

        progressStartAngle = typedArray.getFloat(R.styleable.ProgressView_progressStartAngle, 0f)
        progressStrokeCapRound = typedArray.getBoolean(R.styleable.ProgressView_progressStrokeCapRound, false)

        progressCounterClockwise = typedArray.getBoolean(R.styleable.ProgressView_progressCounterClockwise, false)
        showProgressOnTip = typedArray.getBoolean(R.styleable.ProgressView_showProgressOnTip, false)
        progressOnTipText = typedArray.getString(R.styleable.ProgressView_progressOnTipText)
        progressOnTipTextColor = typedArray.getColor(R.styleable.ProgressView_progressOnTipTextColor, Color.WHITE)
        progressOnTipTextSize = typedArray.getDimension(R.styleable.ProgressView_progressOnTipTextSize, 0f)

        progressOnTipDrawable = typedArray.getDrawable(R.styleable.ProgressView_progressOnTipDrawable)

        showProgressAsPercentage = typedArray.getBoolean(R.styleable.ProgressView_showProgressAsPercentage, false)
        showProgressOnTipAsPercentage = typedArray.getBoolean(R.styleable.ProgressView_showProgressOnTipAsPercentage, false)

        typedArray.recycle()
    }

    override fun setIncludeFontPadding(b: Boolean) = super.setIncludeFontPadding(false)

    override fun setGravity(gravity: Int) = super.setGravity(Gravity.CENTER)

    override fun onMeasure(w: Int, h: Int) {
        val widthMode = MeasureSpec.getMode(w)
        var widthSize = MeasureSpec.getSize(w)
        val heightMode = MeasureSpec.getMode(h)
        var heightSize = MeasureSpec.getSize(h)

        widthSize = measureDimension(widthSize, widthMode)
        heightSize = measureDimension(heightSize, heightMode)

        val size = Math.min(widthSize, heightSize)
        setMeasuredDimension(size, size)
    }

    private fun measureDimension(value: Int, mode: Int) = when (mode) {
        MeasureSpec.AT_MOST -> Math.min(value, defaultSize)
        MeasureSpec.UNSPECIFIED -> value
        MeasureSpec.EXACTLY -> value
        else -> value
    }

    override fun onDraw(canvas: Canvas) {
        val size = width.toFloat()
        val center = size / 2
        val r = radius
        val arcFrom = center - radius
        val arcTo = size - arcFrom
        val angle = angle

        text = if (text.isNullOrEmpty()) progressFormatter.format(progress, minProgressValue, maxProgressValue) else text

        if (progressBackgroundStyle != BackgroundStyle.NONE) canvas.drawCircle(center, center, r, backgroundPaint)

        progressRectF.set(arcFrom, arcFrom, arcTo, arcTo)
        progressPaint.strokeCap = progressPaintCapStyle

        val useCenter = progressPaint.style != Paint.Style.STROKE
        canvas.drawArc(progressRectF, progressStartAngle, angle, useCenter, progressPaint)

        progressOnTipBitmap?.let { applyProgressOnTipBitmap(canvas, center, r, angle, it) }
        if (showProgressOnTip) applyProgressOnTip(canvas, center, r, angle)

        super.onDraw(canvas)
    }

    private fun drawableToBitmap(drawable: Drawable): Bitmap {
        val width = if (drawable.intrinsicWidth > 0) drawable.intrinsicWidth else 1
        val height = if (drawable.intrinsicHeight > 0) drawable.intrinsicHeight else 1

        val bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888)
        val canvas = Canvas(bitmap)
        drawable.setBounds(0, 0, canvas.width, canvas.height)
        drawable.draw(canvas)
        return bitmap
    }

    private fun applyProgressOnTipBitmap(canvas: Canvas, center: Float, r: Float, angle: Float, bitmap: Bitmap) {
        val radAngle = Math.toRadians(progressStartAngle.toDouble() + angle)
        val x = (center + r * Math.cos(radAngle).toFloat()) - bitmap.width / 2
        val y = (center + r * Math.sin(radAngle).toFloat()) - bitmap.height / 2
        canvas.drawBitmap(bitmap, x, y, progressOnTipDrawablePaint)
    }

    private fun applyProgressOnTip(canvas: Canvas, center: Float, r: Float, angle: Float) {
        progressOnTipPaint.color = progressOnTipTextColor
        val text = progressOnTipText ?: ""
        val textW = progressPaint.strokeWidth / 2
        if (progressOnTipTextSize == 0f) {
            progressOnTipPaint.textSize = textW
            progressOnTipPaint.getTextBounds(text, 0, text.length, progressOnTipBounds)
            val textWidth = progressOnTipBounds.width()
            if (textWidth > textW) progressOnTipPaint.textSize *= textW / textWidth
        } else {
            progressOnTipPaint.textSize = progressOnTipTextSize
        }
        progressOnTipPaint.getTextBounds(text, 0, text.length, progressOnTipBounds)

        val height = (progressOnTipPaint.fontMetrics.descent - progressOnTipPaint.fontMetrics.ascent) / 4

        val radAngle = Math.toRadians(progressStartAngle.toDouble() + angle)
        val x = (center + r * Math.cos(radAngle).toFloat())
        val y = (center + r * Math.sin(radAngle).toFloat()) + height
        canvas.drawText(progressOnTipText, x, y, progressOnTipPaint)
    }

    private var progressPaintCapStyle = Paint.Cap.BUTT
        get() = if (progressStrokeCapRound || showProgressOnTip || progressOnTipBitmap != null) Paint.Cap.ROUND else Paint.Cap.BUTT

    private var radius = 0f
        get() = (width.toFloat() - maxPadding - Math.max(backgroundPaint.strokeWidth, progressPaint.strokeWidth)) / 2

    private val maxPadding get() = Math.max(Math.max(paddingStart, paddingEnd), Math.max(paddingTop, paddingBottom)) * 2

    private var angle: Float = 0f
        get() {
            if (progress < minProgressValue || progress > maxProgressValue) return 0f
            val direction = if (progressCounterClockwise) -1 else 1
            return (360 * progressPercent / 100) * direction
        }

    private var progressPercent = 0f
        get() = (progress - minProgressValue) * 100 / (maxProgressValue - minProgressValue)
}
