package pl.exsio.querydsl.entityql;

import com.querydsl.core.dml.StoreClause;
import com.querydsl.core.types.Path;
import pl.exsio.querydsl.entityql.ex.InvalidArgumentException;

import java.util.ArrayList;
import java.util.List;

/**
 * Parent Class of all Static Models generated by EntityQL. Contains some convenience methods.
 *
 * @param <E> - source Entity Type
 */
public class QStaticModel<E> extends QBase<E> {

    protected List<? extends Path<?>> primaryKeyColumns = new ArrayList<>();

    public QStaticModel(Class<? extends E> type, String variable, String schema, String table) {
        super(type, variable, schema, table);
    }



    /**
     * Convenience method used to quiclky set values on insert/update/merge clauses.
     * Method requires the 'params' parameter to be of even size.
     * Every other params array item has to be a String
     *
     * Example:
     *
     * Q<Book> book = qEntity(Book);
     * book.set(
     *     queryFactory.insert(book),
     *     "id", 11L,
     *     "name", "newBook2",
     *     "price", BigDecimal.ONE
     * ).execute();
     *
     * @param clause - insert/update/merge clause
     * @param params - varargs array with parameters
     * @return clause from the 1st argument
     */
    @SuppressWarnings(value = "unchecked")
    public <C extends StoreClause<C>> StoreClause<C> set(StoreClause<C> clause, Object... params) {
        return super.set(clause, key -> {
            if (!(key instanceof Path)) {
                throw new InvalidArgumentException("Param key has to be Path");
            }
            return (Path<Object>) key;
        }, params);
    }

    /**
     * Convenience method to obtain a corresponding Dynamic Query Model
     *
     * @return Dynamic Q-Model for corresponding Static Model
     */
    @SuppressWarnings(value = "unchecked")
    public Q<E> dynamic() {
        Class<E> type = (Class<E>) getType();
        return EntityQL.qEntity(type);
    }

    /**
     * Convenience method to obtain a corresponding Dynamic Query Model
     * <p>
     * Variables are serving as Table Aliases in generated SQL Statements
     * Default variable is always equal to the table name itself.
     * Custom variable is handy if we want to use the same Table multiple
     * times in the same SQL query.
     *
     * @param variable - custom variable name
     * @return Dynamic Q-Model for corresponding Static Model
     */
    @SuppressWarnings(value = "unchecked")
    public Q<E> dynamic(String variable) {
        Class<E> type = (Class<E>) getType();
        return EntityQL.qEntity(type, variable);
    }
}
