package com.crabshue.commons.validations.path.validators;

import java.nio.file.Files;
import java.nio.file.Path;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import com.crabshue.commons.validations.path.annotations.PathAccessMode;
import com.crabshue.commons.validations.path.annotations.ValidFilePath;

public class FilePathValidator implements ConstraintValidator<ValidFilePath, Path> {

    private PathAccessMode[] accessModes;

    @Override
    public void initialize(ValidFilePath constraintAnnotation) {
        accessModes = constraintAnnotation.accessModes();
    }

    @Override
    public boolean isValid(Path path, ConstraintValidatorContext constraintValidatorContext) {

        if (path != null) {
            if (!Files.exists(path)) {
                setErrorMessage(constraintValidatorContext, "The file '${validatedValue}' does not exist");
                return false;
            }

            if (!Files.isRegularFile(path)) {
                setErrorMessage(constraintValidatorContext, "'${validatedValue}' is not a file");
                return false;
            }

            for (PathAccessMode accessMode : accessModes) {
                if (!accessMode.canAccessInMode(path)) {
                    setErrorMessage(constraintValidatorContext, "The file '${validatedValue}' cannot be accessed in " + accessMode.getAccessMode());
                    return false;
                }
            }
        }

        return true;
    }

    private void setErrorMessage(ConstraintValidatorContext constraintValidatorContext, String message) {
        constraintValidatorContext.disableDefaultConstraintViolation();

        constraintValidatorContext
            .buildConstraintViolationWithTemplate(message)
            .addConstraintViolation();
    }
}
