package com.crabshue.commons.json.serialization;

import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;

import com.crabshue.commons.json.serialization.serializers.JSR310DateTimeSerializer;
import com.crabshue.commons.json.serialization.serializers.JSR310LocalDateDeserializer;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.NonNull;

/**
 * Builder for {@link ObjectMapper}.
 */
public class ObjectMapperBuilder {

    /**
     * Get a new instance of the builder.s
     *
     * @return a builder.
     */
    public static ObjectMapperBuilder builder() {

        return new ObjectMapperBuilder();
    }

    private List<JsonInclude.Include> serializationInclusions = new ArrayList<>();

    private ObjectMapperBuilder() {
    }

    /**
     * Configure the {@link ObjectMapper} with a {@link JsonInclude.Include serialization inclusion rule}.
     *
     * @param serializationInclusion the serialization inclusion rule.
     * @return the builder.
     */
    public ObjectMapperBuilder withSerializationInclusion(@NonNull final JsonInclude.Include serializationInclusion) {

        this.serializationInclusions.add(serializationInclusion);
        return this;
    }

    /**
     * Build the configured {@link ObjectMapper}.
     *
     * @return the object mapper.o
     */
    public ObjectMapper build() {

        final ObjectMapper ret = new ObjectMapper();

        final JavaTimeModule module = new JavaTimeModule();
        module.addSerializer(OffsetDateTime.class, JSR310DateTimeSerializer.INSTANCE);
        module.addSerializer(ZonedDateTime.class, JSR310DateTimeSerializer.INSTANCE);
        module.addSerializer(LocalDateTime.class, JSR310DateTimeSerializer.INSTANCE);
        module.addSerializer(Instant.class, JSR310DateTimeSerializer.INSTANCE);
        module.addDeserializer(LocalDate.class, JSR310LocalDateDeserializer.INSTANCE);

        ret.registerModule(module);

        this.serializationInclusions.forEach(ret::setSerializationInclusion);

        return ret;
    }
}
