package com.crabshue.commons.json.serialization;

import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collection;

import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.exceptions.context.CommonErrorContext;
import com.crabshue.commons.file.FileSystemUtils;
import com.crabshue.commons.file.exceptions.FileErrorContext;
import com.crabshue.commons.file.exceptions.FileErrorType;
import com.crabshue.commons.json.serialization.exceptions.JsonErrorType;
import com.crabshue.commons.json.serialization.serializers.JSR310DateTimeSerializer;
import com.crabshue.commons.json.serialization.serializers.JSR310LocalDateDeserializer;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import lombok.NonNull;

/**
 * Utility class for JSON serialization/deserialization.
 */
public class JsonSerializationUtils {

    /**
     * Convert an object to its JSON representation.
     *
     * @param object the object.
     * @return the JSON representation.
     */
    public static String convertObjectToJsonString(@NonNull final Object object) {
        final ObjectMapper mapper = provideObjectMapper();

        try {
            return mapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new SystemException(JsonErrorType.JSON_CONVERSION_ERROR, e)
                .addContextValue(CommonErrorContext.CAUSE, object);
        }
    }

    /**
     * Convert an object to its JSON representation and serialize it in the given file.
     *
     * @param object the object.
     * @param file   the file.
     * @return the JSON representation serialized file.
     */
    public static File convertObjectToJsonFile(@NonNull final Object object,
                                               @NonNull final File file) {

        FileSystemUtils.retrieveOrCreateFile(file);

        final ObjectMapper mapper = provideObjectMapper();
        try {
            mapper.writeValue(file, object);
        } catch (IOException e) {
            throw new SystemException(FileErrorType.ERROR_WRITING_FILE, e)
                .addContextValue(CommonErrorContext.CAUSE, object)
                .addContextValue(FileErrorContext.FILE, file);
        }

        return file;
    }


    /**
     * Deserialize an object from its JSON file representation.
     *
     * @param file  the file.
     * @param clazz the target class to deserialize to.
     * @param <T>   the return type.
     * @return the deserialized object.
     */
    public static <T> T readJsonFileToObject(@NonNull final File file,
                                             @NonNull final Class<T> clazz) {

        final ObjectMapper mapper = provideObjectMapper();
        try {
            return mapper.readValue(file, clazz);
        } catch (IOException e) {
            throw new SystemException(JsonErrorType.JSON_PARSING_ERROR, e)
                .addContextValue(CommonErrorContext.CAUSE, file);
        }
    }

    /**
     * Deserialize a collection object from its JSON file representation.
     *
     * @param file       the file.
     * @param arrayClazz the array version of the target class to deserialize to.
     * @param <T>        the return type.
     * @return the deserialized object.
     */
    public static <T> Collection<T> readJsonFileToCollection(@NonNull final File file,
                                                             @NonNull final Class<T[]> arrayClazz) {

        final ObjectMapper mapper = provideObjectMapper();
        try {
            return Arrays.asList(mapper.readValue(file, arrayClazz));
        } catch (IOException e) {
            throw new SystemException(JsonErrorType.JSON_PARSING_ERROR, e)
                .addContextValue(CommonErrorContext.CAUSE, file);
        }
    }

    private static ObjectMapper provideObjectMapper() {

        return ObjectMapperHolder.instance;
    }

    private static class ObjectMapperHolder {

        static ObjectMapper instance;

        static {
            instance = new ObjectMapper();

            JavaTimeModule module = new JavaTimeModule();
            module.addSerializer(OffsetDateTime.class, JSR310DateTimeSerializer.INSTANCE);
            module.addSerializer(ZonedDateTime.class, JSR310DateTimeSerializer.INSTANCE);
            module.addSerializer(LocalDateTime.class, JSR310DateTimeSerializer.INSTANCE);
            module.addSerializer(Instant.class, JSR310DateTimeSerializer.INSTANCE);
            module.addDeserializer(LocalDate.class, JSR310LocalDateDeserializer.INSTANCE);
            instance.registerModule(module);

        }
    }
}
