package com.crabshue.commons.json.jsonpath;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Optional;

import com.crabshue.commons.exceptions.ApplicationException;
import com.crabshue.commons.exceptions.SystemException;
import com.crabshue.commons.json.jsonpath.exceptions.JsonPathErrorContext;
import com.crabshue.commons.json.jsonpath.exceptions.JsonPathErrorType;
import com.crabshue.commons.json.serialization.exceptions.JsonErrorType;
import com.jayway.jsonpath.DocumentContext;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.JsonPathException;
import com.jayway.jsonpath.PathNotFoundException;
import lombok.NonNull;
import lombok.extern.slf4j.Slf4j;

/**
 * JSON Path evaluator.
 */
@Slf4j
public class JsonPathEvaluator {

    private DocumentContext jsonObject;

    private String jsonPath;

    public static JsonPathEvaluator of(@NonNull final String json) {

        final JsonPathEvaluator ret = new JsonPathEvaluator();
        ret.jsonObject = JsonPath.parse(json);

        return ret;
    }

    public static JsonPathEvaluator of(@NonNull final InputStream json) {

        final JsonPathEvaluator ret = new JsonPathEvaluator();
        ret.jsonObject = JsonPath.parse(json);

        return ret;
    }

    public static JsonPathEvaluator of(@NonNull final File json) {

        final JsonPathEvaluator ret = new JsonPathEvaluator();
        try {
            ret.jsonObject = JsonPath.parse(json);
        } catch (IOException e) {
            throw new SystemException(JsonErrorType.JSON_PARSING_ERROR, e)
                .addContextValue(JsonPathErrorContext.JSON_FILE, json);
        }

        return ret;
    }

    public static JsonPathEvaluator of(@NonNull final URL json) {

        final JsonPathEvaluator ret = new JsonPathEvaluator();
        try {
            ret.jsonObject = JsonPath.parse(json);
        } catch (IOException e) {
            throw new SystemException(JsonErrorType.JSON_PARSING_ERROR, e)
                .addContextValue(JsonPathErrorContext.JSON_URL, json);
        }
        return ret;
    }

    public JsonPathEvaluator withJsonPath(@NonNull final String jsonPath) {

        this.jsonPath = jsonPath;
        return this;
    }

    public <T> T evaluate() {
        try {
            final T ret = this.jsonObject.read(this.jsonPath);
            logger.info("Json path evaluated [{}] : [{}]", this.jsonPath, ret);
            return ret;
        } catch (JsonPathException e) {
            throw new ApplicationException(JsonPathErrorType.JSON_PATH_EVALUATION_ERROR, e)
                .addContextValue(JsonPathErrorContext.JSON, jsonObject)
                .addContextValue(JsonPathErrorContext.JSON_PATH, this.jsonPath);
        }
    }

    public <T> T evaluate(Class<T> clazz) {
        try {
            final T ret = this.jsonObject.read(this.jsonPath, clazz);
            logger.info("Json path evaluated [{}] : [{}]", this.jsonPath, ret);
            return ret;
        } catch (JsonPathException e) {
            throw new ApplicationException(JsonPathErrorType.JSON_PATH_EVALUATION_ERROR, e)
                .addContextValue(JsonPathErrorContext.JSON, jsonObject)
                .addContextValue(JsonPathErrorContext.JSON_PATH, this.jsonPath);
        }
    }

    public <T> Optional<T> evaluateOptional() {
        try {
            final T ret = this.jsonObject.read(this.jsonPath);
            logger.info("Json path evaluated [{}] : [{}]", this.jsonPath, ret);
            return Optional.ofNullable(ret);

        } catch (PathNotFoundException e) {
            logger.warn("No result found for JSON path [{}]", this.jsonPath);
            return Optional.empty();
        } catch (JsonPathException e) {
            throw new ApplicationException(JsonPathErrorType.JSON_PATH_EVALUATION_ERROR, e)
                .addContextValue(JsonPathErrorContext.JSON, jsonObject)
                .addContextValue(JsonPathErrorContext.JSON_PATH, this.jsonPath);
        }
    }
}
