package com.crabshue.commons.exceptions;

import org.apache.commons.lang3.exception.ContextedRuntimeException;

import com.crabshue.commons.exceptions.context.CommonErrorContext;
import com.crabshue.commons.exceptions.context.ErrorContext;
import com.crabshue.commons.exceptions.context.ErrorType;
import com.crabshue.commons.exceptions.utils.ExceptionUtils;


/**
 * Abstract exception.
 */
public abstract class AbstractException extends ContextedRuntimeException {

    protected AbstractException() {
    }

    /**
     * Constructor with an {@link ErrorType}.
     *
     * @param errorType the error type.
     */
    public AbstractException(final ErrorType errorType) {
        super(errorType.getMessage());
        this.addContextValue(CommonErrorContext.EXCEPTION_MESSAGE, errorType);
    }

    /**
     * Constructor with an {@link ErrorType} and a {@link Throwable}.
     *
     * @param errorType the error type.
     * @param cause     the cause of the exception.
     */
    public AbstractException(final ErrorType errorType, final Throwable cause) {
        super(errorType.getMessage(), cause);
        this.addContextValue(CommonErrorContext.EXCEPTION_MESSAGE, errorType);
        this.addContextValue(CommonErrorContext.CAUSE, ExceptionUtils.buildMessageFromThrowable(cause));
    }

    /**
     * Constructor with an {@link ErrorType} and a custom message.
     *
     * @param errorType the error type.
     * @param message   the custom message.
     */
    public AbstractException(final ErrorType errorType, final String message) {
        super(message);
        this.addContextValue(CommonErrorContext.EXCEPTION_MESSAGE, errorType);
    }

    /**
     * Constructor with an {@link ErrorType}, a custom message, and a {@link Throwable}.
     *
     * @param errorType the error type.
     * @param message   the custom message.
     * @param cause     the cause of the exception.
     */
    public AbstractException(final ErrorType errorType, final String message, final Throwable cause) {
        super(message, cause);
        this.addContextValue(CommonErrorContext.EXCEPTION_MESSAGE, errorType);
        this.addContextValue(CommonErrorContext.CAUSE, ExceptionUtils.buildMessageFromThrowable(cause));
    }

    /**
     * Add a context entry to an exception as a label and an object.
     *
     * @param label the label.
     * @param value the value.
     */
    @Override
    public AbstractException addContextValue(final String label, final Object value) {
        return (AbstractException) super.addContextValue(label, value);
    }

    /**
     * Add a context entry to an exception as an {@link ErrorContext} and an object.
     *
     * @param label the error context.
     * @param value the value.
     */
    public AbstractException addContextValue(final ErrorContext label, final Object value) {
        return (AbstractException) super.addContextValue(label.name(), value);
    }

}
