/*
 * Decompiled with CFR 0.152.
 */
package com.innovatrics.mrz;

import com.innovatrics.mrz.MrzParseException;
import com.innovatrics.mrz.MrzRange;
import com.innovatrics.mrz.MrzRecord;
import com.innovatrics.mrz.types.MrzDate;
import com.innovatrics.mrz.types.MrzFormat;
import com.innovatrics.mrz.types.MrzSex;
import java.text.Normalizer;
import java.util.HashMap;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MrzParser {
    private static final Logger LOG = LoggerFactory.getLogger(MrzParser.class);
    private static final int[] MRZ_WEIGHTS = new int[]{7, 3, 1};
    private static final Map<String, String> EXPAND_CHARACTERS = new HashMap<String, String>();
    public static final char FILLER = '<';
    private final String mrz;
    private final String[] rows;
    private final MrzFormat format;

    public MrzParser(String mrz) throws MrzParseException {
        this.mrz = mrz;
        this.rows = mrz.split("\n");
        this.format = MrzFormat.get(mrz);
    }

    public String getMrz() {
        return this.mrz;
    }

    public String[] getRows() {
        return this.rows;
    }

    public MrzFormat getFormat() {
        return this.format;
    }

    public String[] parseName(MrzRange range) throws MrzParseException {
        this.checkValidCharacters(range);
        String str = this.rawValue(range);
        while (str.endsWith("<")) {
            str = str.substring(0, str.length() - 1);
        }
        String[] names = str.split("<<");
        String givenNames = "";
        String surname = this.parseString(new MrzRange(range.getColumn(), range.getColumn() + names[0].length(), range.getRow()));
        if (names.length == 1) {
            givenNames = this.parseString(new MrzRange(range.getColumn(), range.getColumn() + names[0].length(), range.getRow()));
            surname = "";
        } else if (names.length > 1) {
            surname = this.parseString(new MrzRange(range.getColumn(), range.getColumn() + names[0].length(), range.getRow()));
            givenNames = this.parseString(new MrzRange(range.getColumn() + names[0].length() + 2, range.getColumn() + str.length(), range.getRow()));
        }
        return new String[]{surname, givenNames};
    }

    public String rawValue(MrzRange ... range) {
        StringBuilder sb = new StringBuilder();
        for (MrzRange r : range) {
            sb.append(this.getRows()[r.getRow()].substring(r.getColumn(), r.getColumnTo()));
        }
        return sb.toString();
    }

    public void checkValidCharacters(MrzRange range) throws MrzParseException {
        String str = this.rawValue(range);
        for (int i = 0; i < str.length(); ++i) {
            char c = str.charAt(i);
            if (c == '<' || c >= '0' && c <= '9' || c >= 'A' && c <= 'Z') continue;
            throw new MrzParseException("Invalid character in MRZ record: " + c, this.getMrz(), new MrzRange(range.getColumn() + i, range.getColumn() + i + 1, range.getRow()), this.getFormat());
        }
    }

    public String parseString(MrzRange range) throws MrzParseException {
        this.checkValidCharacters(range);
        String str = this.rawValue(range);
        while (str.endsWith("<")) {
            str = str.substring(0, str.length() - 1);
        }
        return str.replace("<<", ", ").replace('<', ' ');
    }

    public boolean checkDigit(int col, int row, MrzRange strRange, String fieldName) {
        return this.checkDigit(col, row, this.rawValue(strRange), fieldName);
    }

    public boolean checkDigit(int col, int row, String str, String fieldName) {
        MrzRange invalidCheckdigit = null;
        char digit = (char)(MrzParser.computeCheckDigit(str) + 48);
        char checkDigit = this.getRows()[row].charAt(col);
        if (checkDigit == '<') {
            checkDigit = '0';
        }
        if (digit != checkDigit) {
            invalidCheckdigit = new MrzRange(col, col + 1, row);
            LOG.info("Check digit verification failed for " + fieldName + ": expected " + digit + " but got " + checkDigit);
        }
        return invalidCheckdigit == null;
    }

    public MrzDate parseDate(MrzRange range) {
        if (range.length() != 6) {
            throw new IllegalArgumentException("Parameter range: invalid value " + range + ": must be 6 characters long");
        }
        int year = this.parseYear(range);
        int month = this.parseMonth(range);
        int day = this.parseDay(range);
        return new MrzDate(year, month, day, this.rawValue(range));
    }

    private int parseYear(MrzRange range) {
        int year;
        MrzRange r = new MrzRange(range.getColumn(), range.getColumn() + 2, range.getRow());
        try {
            year = Integer.parseInt(this.rawValue(r));
        }
        catch (NumberFormatException ex) {
            year = -1;
            LOG.debug("Failed to parse MRZ date year " + this.rawValue(range) + ": " + ex, (Object)this.getMrz(), (Object)r);
        }
        if (year < 0 || year > 99) {
            LOG.debug("Invalid year value " + year + ": must be 0..99");
        }
        return year;
    }

    private int parseMonth(MrzRange range) {
        int month;
        MrzRange r = new MrzRange(range.getColumn() + 2, range.getColumn() + 4, range.getRow());
        try {
            month = Integer.parseInt(this.rawValue(r));
        }
        catch (NumberFormatException ex) {
            month = -1;
            LOG.debug("Failed to parse MRZ date month " + this.rawValue(range) + ": " + ex, (Object)this.getMrz(), (Object)r);
        }
        if (month < 1 || month > 12) {
            LOG.debug("Invalid month value " + month + ": must be 1..12");
        }
        return month;
    }

    private int parseDay(MrzRange range) {
        int day;
        MrzRange r = new MrzRange(range.getColumn() + 4, range.getColumn() + 6, range.getRow());
        try {
            day = Integer.parseInt(this.rawValue(r));
        }
        catch (NumberFormatException ex) {
            day = -1;
            LOG.debug("Failed to parse MRZ date day " + this.rawValue(range) + ": " + ex, (Object)this.getMrz(), (Object)r);
        }
        if (day < 1 || day > 31) {
            LOG.debug("Invalid day value " + day + ": must be 1..31");
        }
        return day;
    }

    public MrzSex parseSex(int col, int row) {
        return MrzSex.fromMrz(this.getRows()[row].charAt(col));
    }

    private static boolean isValid(char c) {
        return c == '<' || c >= '0' && c <= '9' || c >= 'A' && c <= 'Z';
    }

    private static int getCharacterValue(char c) {
        if (c == '<') {
            return 0;
        }
        if (c >= '0' && c <= '9') {
            return c - 48;
        }
        if (c >= 'A' && c <= 'Z') {
            return c - 65 + 10;
        }
        throw new IllegalArgumentException("Invalid character in MRZ record: " + c);
    }

    public static int computeCheckDigit(String str) {
        int result = 0;
        for (int i = 0; i < str.length(); ++i) {
            result += MrzParser.getCharacterValue(str.charAt(i)) * MRZ_WEIGHTS[i % MRZ_WEIGHTS.length];
        }
        return result % 10;
    }

    public static char computeCheckDigitChar(String str) {
        return (char)(48 + MrzParser.computeCheckDigit(str));
    }

    public static MrzRecord parse(String mrz) throws MrzParseException {
        MrzRecord result = MrzFormat.get(mrz).newRecord();
        result.fromMrz(mrz);
        return result;
    }

    public static String toMrz(String input, int length) {
        String string = input == null ? "" : input;
        for (Map.Entry<String, String> e : EXPAND_CHARACTERS.entrySet()) {
            string = string.replace(e.getKey(), e.getValue());
        }
        string = string.replace("\u2019", "");
        string = string.replace("'", "");
        string = MrzParser.deaccent(string).toUpperCase();
        if (length >= 0 && string.length() > length) {
            string = string.substring(0, length);
        }
        StringBuilder sb = new StringBuilder(string);
        for (int i = 0; i < sb.length(); ++i) {
            if (MrzParser.isValid(sb.charAt(i))) continue;
            sb.setCharAt(i, '<');
        }
        while (sb.length() < length) {
            sb.append('<');
        }
        return sb.toString();
    }

    private static boolean isBlank(String str) {
        return str == null || str.trim().length() == 0;
    }

    public static String nameToMrz(String surname, String givenNames, int length) {
        if (MrzParser.isBlank(surname)) {
            throw new IllegalArgumentException("Parameter surname: invalid value " + surname + ": blank");
        }
        if (MrzParser.isBlank(givenNames)) {
            throw new IllegalArgumentException("Parameter givenNames: invalid value " + givenNames + ": blank");
        }
        if (length <= 0) {
            throw new IllegalArgumentException("Parameter length: invalid value " + length + ": not positive");
        }
        String[] surnames = MrzParser.extractNames(surname);
        String[] given = MrzParser.extractNames(givenNames);
        MrzParser.truncateNames(surnames, given, length, surname, givenNames);
        return MrzParser.toMrz(MrzParser.toName(surnames, given), length);
    }

    private static String[] extractNames(String name) {
        String[] names = name.replace(", ", " ").trim().split("[ \n\t\f\r]+");
        for (int i = 0; i < names.length; ++i) {
            names[i] = MrzParser.toMrz(names[i], -1);
        }
        return names;
    }

    private static void truncateNames(String[] surnames, String[] given, int length, String surname, String givenNames) {
        int nameSize = MrzParser.getNameSize(surnames, given);
        String[] currentlyTruncating = given;
        int currentlyTruncatingIndex = given.length - 1;
        while (nameSize > length) {
            String ct = currentlyTruncating[currentlyTruncatingIndex];
            int ctsize = ct.length();
            if (nameSize - ctsize + 1 <= length) {
                currentlyTruncating[currentlyTruncatingIndex] = ct.substring(0, ctsize - (nameSize - length));
            } else {
                currentlyTruncating[currentlyTruncatingIndex] = ct.substring(0, 1);
                if (--currentlyTruncatingIndex < 0) {
                    if (currentlyTruncating == surnames) {
                        throw new IllegalArgumentException("Cannot truncate name " + surname + " " + givenNames + ": length too small: " + length + "; truncated to " + MrzParser.toName(surnames, given));
                    }
                    currentlyTruncating = surnames;
                    currentlyTruncatingIndex = currentlyTruncating.length - 1;
                }
            }
            nameSize = MrzParser.getNameSize(surnames, given);
        }
    }

    private static String toName(String[] surnames, String[] given) {
        StringBuilder sb = new StringBuilder();
        boolean first = true;
        for (String s : surnames) {
            if (first) {
                first = false;
            } else {
                sb.append('<');
            }
            sb.append(s);
        }
        sb.append('<');
        for (String s : given) {
            sb.append('<');
            sb.append(s);
        }
        return sb.toString();
    }

    private static int getNameSize(String[] surnames, String[] given) {
        int result = 0;
        for (String s : surnames) {
            result += s.length() + 1;
        }
        for (String s : given) {
            result += s.length() + 1;
        }
        return result;
    }

    private static String deaccent(String str) {
        String n = Normalizer.normalize(str, Normalizer.Form.NFD);
        return n.replaceAll("[^\\p{ASCII}]", "").toLowerCase();
    }

    static {
        EXPAND_CHARACTERS.put("\u00c4", "AE");
        EXPAND_CHARACTERS.put("\u00e4", "AE");
        EXPAND_CHARACTERS.put("\u00c5", "AA");
        EXPAND_CHARACTERS.put("\u00e5", "AA");
        EXPAND_CHARACTERS.put("\u00c6", "AE");
        EXPAND_CHARACTERS.put("\u00e6", "AE");
        EXPAND_CHARACTERS.put("\u0132", "IJ");
        EXPAND_CHARACTERS.put("\u0133", "IJ");
        EXPAND_CHARACTERS.put("\u00d6", "OE");
        EXPAND_CHARACTERS.put("\u00f6", "OE");
        EXPAND_CHARACTERS.put("\u00d8", "OE");
        EXPAND_CHARACTERS.put("\u00f8", "OE");
        EXPAND_CHARACTERS.put("\u00dc", "UE");
        EXPAND_CHARACTERS.put("\u00fc", "UE");
        EXPAND_CHARACTERS.put("\u00df", "SS");
    }
}

