/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.iceberg.types;

import com.google.common.base.Charsets;
import com.netflix.iceberg.exceptions.RuntimeIOException;
import com.netflix.iceberg.types.Type;
import com.netflix.iceberg.types.Types;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.Arrays;
import java.util.UUID;

public class Conversions {
    private static final String HIVE_NULL = "__HIVE_DEFAULT_PARTITION__";
    private static final ThreadLocal<CharsetEncoder> ENCODER = ThreadLocal.withInitial(Charsets.UTF_8::newEncoder);
    private static final ThreadLocal<CharsetDecoder> DECODER = ThreadLocal.withInitial(Charsets.UTF_8::newDecoder);

    public static Object fromPartitionString(Type type, String asString) {
        if (asString == null || HIVE_NULL.equals(asString)) {
            return null;
        }
        switch (type.typeId()) {
            case BOOLEAN: {
                return Boolean.valueOf(asString);
            }
            case INTEGER: {
                return Integer.valueOf(asString);
            }
            case LONG: {
                return Long.valueOf(asString);
            }
            case FLOAT: {
                return Long.valueOf(asString);
            }
            case DOUBLE: {
                return Double.valueOf(asString);
            }
            case STRING: {
                return asString;
            }
            case UUID: {
                return UUID.fromString(asString);
            }
            case FIXED: {
                Types.FixedType fixed = (Types.FixedType)type;
                return Arrays.copyOf(asString.getBytes(Charsets.UTF_8), fixed.length());
            }
            case BINARY: {
                return asString.getBytes(Charsets.UTF_8);
            }
            case DECIMAL: {
                return new BigDecimal(asString);
            }
        }
        throw new UnsupportedOperationException("Unsupported type for fromPartitionString: " + type);
    }

    public static ByteBuffer toByteBuffer(Type type, Object value) {
        switch (type.typeId()) {
            case BOOLEAN: {
                return ByteBuffer.allocate(1).put(0, (Boolean)value != false ? (byte)1 : 0);
            }
            case INTEGER: 
            case DATE: {
                return ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putInt(0, (Integer)value);
            }
            case LONG: 
            case TIME: 
            case TIMESTAMP: {
                return ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN).putLong(0, (Long)value);
            }
            case FLOAT: {
                return ByteBuffer.allocate(4).order(ByteOrder.LITTLE_ENDIAN).putFloat(0, ((Float)value).floatValue());
            }
            case DOUBLE: {
                return ByteBuffer.allocate(8).order(ByteOrder.LITTLE_ENDIAN).putDouble(0, (Double)value);
            }
            case STRING: {
                CharBuffer buffer = CharBuffer.wrap((CharSequence)value);
                try {
                    return ENCODER.get().encode(buffer);
                }
                catch (CharacterCodingException e) {
                    throw new RuntimeIOException(e, "Failed to encode value as UTF-8: " + value, new Object[0]);
                }
            }
            case UUID: {
                UUID uuid = (UUID)value;
                return ByteBuffer.allocate(16).order(ByteOrder.LITTLE_ENDIAN).putLong(0, uuid.getMostSignificantBits()).putLong(1, uuid.getLeastSignificantBits());
            }
            case FIXED: 
            case BINARY: {
                return (ByteBuffer)value;
            }
            case DECIMAL: {
                return ByteBuffer.wrap(((BigDecimal)value).unscaledValue().toByteArray());
            }
        }
        throw new UnsupportedOperationException("Cannot serialize type: " + type);
    }

    public static <T> T fromByteBuffer(Type type, ByteBuffer buffer) {
        return (T)Conversions.internalFromByteBuffer(type, buffer);
    }

    private static Object internalFromByteBuffer(Type type, ByteBuffer buffer) {
        ByteBuffer tmp = buffer.duplicate().order(ByteOrder.LITTLE_ENDIAN);
        switch (type.typeId()) {
            case BOOLEAN: {
                return tmp.get() != 0;
            }
            case INTEGER: 
            case DATE: {
                return tmp.getInt();
            }
            case LONG: 
            case TIME: 
            case TIMESTAMP: {
                if (tmp.remaining() < 8) {
                    return (long)tmp.getInt();
                }
                return tmp.getLong();
            }
            case FLOAT: {
                return Float.valueOf(tmp.getFloat());
            }
            case DOUBLE: {
                if (tmp.remaining() < 8) {
                    return (double)tmp.getFloat();
                }
                return tmp.getDouble();
            }
            case STRING: {
                try {
                    return DECODER.get().decode(tmp);
                }
                catch (CharacterCodingException e) {
                    throw new RuntimeIOException(e, "Failed to decode value as UTF-8: " + buffer, new Object[0]);
                }
            }
            case UUID: {
                long mostSigBits = tmp.getLong();
                long leastSigBits = tmp.getLong();
                return new UUID(mostSigBits, leastSigBits);
            }
            case FIXED: 
            case BINARY: {
                return tmp;
            }
            case DECIMAL: {
                Types.DecimalType decimal = (Types.DecimalType)type;
                byte[] unscaledBytes = new byte[buffer.remaining()];
                tmp.get(unscaledBytes);
                return new BigDecimal(new BigInteger(unscaledBytes), decimal.scale());
            }
        }
        throw new UnsupportedOperationException("Cannot deserialize type: " + type);
    }
}

