package com.flybits.android.push

import android.content.Context
import android.content.Context.MODE_PRIVATE
import android.content.SharedPreferences
import android.os.Handler
import android.os.Looper
import androidx.work.ExistingPeriodicWorkPolicy
import androidx.work.PeriodicWorkRequest
import androidx.work.WorkManager
import com.flybits.android.push.db.PushDatabase
import com.flybits.android.push.worker.PushWorker
import com.flybits.commons.library.SharedElementsFactory
import com.flybits.commons.library.api.FlybitsScope
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.User
import com.flybits.commons.library.utils.Utilities
import com.flybits.internal.db.CommonsDatabase
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.*
import java.util.concurrent.Executors
import java.util.concurrent.TimeUnit


/**
 * This constructor should be used in the event that the application subscribed to Firebase
 * specific push notifications.
 */
const val TAG = "PushScope"

class PushScope internal constructor(var context: Context? = null) : FlybitsScope("PushScope") {

    private var properties: HashMap<String, String>? = null
    private var autoConnectToPush: Boolean = true

    /**
     * Constructor accepts properties will trigger onStart() and onStop() but not its operations.
     */
    constructor(properties: HashMap<String, String>) : this() {
        this.properties = properties
    }

    /**
     * Constructor accepts properties and context will trigger onStart() and onStop() operations.
     */
    constructor(properties: HashMap<String, String>, context: Context) : this(context) {
        this.properties = properties
    }

    /**
     * Constructor accepts autoConnect trigger onStart() and onStop() but not  its operations.
     */
    constructor(autoConnect: Boolean) : this() {
        autoConnectToPush = autoConnect
    }

    /**
     * Constructor accepts autoConnect and context will trigger onStart() and onStop() operations.
     */
    constructor(autoConnect: Boolean, context: Context) : this(context) {
        autoConnectToPush = autoConnect
    }

    override fun onConnected(context: Context, user: User) {
        if (autoConnectToPush && user.isOptedIn) {
            PushManager.enableFCMPush(context, properties, null)
            schedulePushWorkers()
        }
        PushManager.setLanguage(context, Utilities.getDeviceLanguageCodes(context))
    }

    override fun onDisconnected(context: Context, jwtToken: String) {
        clearDataAsync(context)
        cancelWorkers()
    }

    override fun onStart() {
        // Implement onStart for Push
        context?.let {
            CoroutineScope(Dispatchers.Default).launch {
                val isJwt = SharedElementsFactory.get(it).getSavedJWTToken()
                val user = CommonsDatabase.getDatabase(it).userDao().activeUser
                user?.let { userActive ->
                    if (isJwt.isNotEmpty()) {
                        if (autoConnectToPush && userActive.isOptedIn) {
                            PushManager.enableFCMPush(it, properties, null)
                            schedulePushWorkers()
                        }
                        PushManager.setLanguage(it, Utilities.getDeviceLanguageCodes(it))
                    }
                } ?: Logger.appendTag(TAG).e("User does not Exist")
            }
        }
    }

    override fun onStop() {
        // Implement onStop for Push
        context?.let {
            val jwt = SharedElementsFactory.get(it).getSavedJWTToken()
            if (jwt.isNotEmpty()) {
                disableThenClearData(it, jwt, Handler(Looper.getMainLooper()))
                cancelWorkers()
            }
        }
    }

    override fun onAccountDestroyed(context: Context, jwtToken: String) {
        disableThenClearData(context, jwtToken, Handler(Looper.getMainLooper()))
    }

    override fun onOptedStateChange(context: Context, optedState: Boolean) {
        if (!optedState) {
            clearDataAsync(context)
        } else {
            if (autoConnectToPush) {
                PushManager.enableFCMPush(context, properties, null)
            }
            PushManager.setLanguage(context, Utilities.getDeviceLanguageCodes(context))
        }
    }

    /**
     * Disable push if autoConnectToPush is true, then clear all databases.
     *
     * @param context Context associated to the appliation.
     * @param jwt JWT token associated with current session.
     * @param handler Handler that the callback results will be posted on.
     */
    internal fun disableThenClearData(context: Context, jwt: String, handler: Handler) {
        if (autoConnectToPush) {
            PushManager.disablePush(context, jwt, object : BasicResultCallback {
                override fun onSuccess() {
                    clearDataAsync(context)
                }

                override fun onException(exception: FlybitsException) {}
            }, handler)
        }
    }

    /**
     * Clear the kernel databases on the worker thread.
     *
     * @param context Context associated with the application.
     */
    internal fun clearDataAsync(context: Context) {
        Executors.newSingleThreadExecutor().execute { clearData(context) }
    }

    /**
     * Clear the kernel databases. Warning, this method will run on the caller thread.
     * See [PushScope.clearDataAsync] for asynchronous alternative.
     *
     * @param context Context associated with the application.
     */
    internal fun clearData(context: Context) {
        val preferences = getPushPreferences(context).edit()
        preferences.clear()
        preferences.apply()
        PushDatabase.getDatabase(context).pushDataDAO().clear()
    }

    private fun schedulePushWorkers() {
        val taskGetPluginsRepeat =
            PeriodicWorkRequest.Builder(
                PushWorker::class.java,
                NOTIFICATION__WORKER_INTERVAl,
                TimeUnit.DAYS
            )
                .addTag(NOTIFICATION__WORKER)
                .build()
        WorkManager.getInstance().enqueueUniquePeriodicWork(
            NOTIFICATION__WORKER,
            ExistingPeriodicWorkPolicy.KEEP,
            taskGetPluginsRepeat
        )
    }

    private fun cancelWorkers() {
        WorkManager.getInstance().cancelUniqueWork(NOTIFICATION__WORKER)
    }

    companion object {
        const val NOTIFICATION__WORKER_INTERVAl = 1L
        const val PUSH_PREF = "FLYBITS_PREF_PUSH"
        const val NOTIFICATION__WORKER = "WORKER_NAME_PUSH_NOTIFICATION"
        const val ROOT = "/push"
        private var context: Context? = null

        /**
         * Empty Constructor for initializing PushScope.
         *
         */
        @JvmField
        var SCOPE = PushScope()

        /**
         * Constructor for initializing PushScope with Context.
         */
        @JvmStatic
        fun get(context: Context): PushScope {
            this.context = context
            SCOPE = PushScope(this.context)
            return SCOPE
        }

        /**
         * Get the default shared preferences for the application.
         *
         * @param context The context of the application.
         * @return The default SharedPreferences object for the application.
         */
        @JvmStatic
        fun getPushPreferences(context: Context): SharedPreferences {
            return context.getSharedPreferences(PUSH_PREF, MODE_PRIVATE)
        }
    }
}
