package com.flybits.android.push.analytics

import android.content.Context
import com.flybits.android.push.models.Push
import com.flybits.android.push.models.newPush.DisplayablePush
import com.flybits.commons.library.analytics.Analytics

/**
 * This class is responsible for tracking Push analytics. Example of this include:
 * push viewed, push engaged, or push component engaged.
 *
 * @param context The context of the application.
 */
class PushAnalytics(context: Context): Analytics(context) {

    companion object {
        const val analyticsScope = "pushAnalytics"
    }

    private enum class Action(val actionName: String){
        VIEWED("viewed")
        , ENGAGED("engaged")
        , COMPONENT_ENGAGED("componentEngaged")
    }

    private fun track(action: Action, pushRequestId: String, timestamp: Long){
        super.track(AnalyticsEvent(analyticsScope, "query.${action.actionName}.$pushRequestId", true, timestamp))
    }

    /**
     * Track a push item being viewed using a [Push] object.
     *
     * @param push [Push] being viewed.
     * @param timestamp Timestamp of the event.
     */
    @Deprecated("Deprecated in version 1.19.0, will be removed in version 3.0.0",
        ReplaceWith("trackViewed(push, timestamp)", "com.flybits.android.push.analytics"),
        DeprecationLevel.WARNING)
    fun trackViewed(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, push.requestId, timestamp)
    }

    /**
     * Track a push item being viewed using a [DisplayablePush] object.
     *
     * @param push [DisplayablePush] being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackViewed(push: DisplayablePush, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, push.parentId, timestamp)
    }

    /**
     * Track a push being viewed using the push requestId.
     *
     * @param pushRequestId The requestId of the [Push] being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackViewed(pushRequestId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.VIEWED, pushRequestId, timestamp)
    }

    /**
     * Track a push being engaged using a [Push] object.
     *
     * @param push [Push] being engaged.
     * @param timestamp Timestamp of the event.
     *
     */
    @Deprecated("Deprecated in version 1.19.0, will be removed in version 3.0.0",
        ReplaceWith("trackEngaged(push, timestamp)", "com.flybits.android.push.analytics"),
        DeprecationLevel.WARNING)
    fun trackEngaged(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, push.requestId, timestamp)
    }

    /**
     * Track a push being engaged using a [DisplayablePush] object.
     *
     * @param push [DisplayablePush] being engaged.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackEngaged(push: DisplayablePush, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, push.parentId, timestamp)
    }

    /**
     * Track a push being engaged using a push requestId.
     *
     * @param pushRequestId The requestId of the [Push] being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackEngaged(pushRequestId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.ENGAGED, pushRequestId, timestamp)
    }

    /**
     * Track a push component being engaged using a [Push] object.
     *
     * @param push [Push] associated with the component engaged event.
     * @param timestamp Timestamp of the event.
     *
     */
    @Deprecated("Deprecated in version 1.19.0, will be removed in version 3.0.0",
        ReplaceWith("trackComponentEngaged(push, timestamp)", "com.flybits.android.push.analytics"),
        DeprecationLevel.WARNING)
    fun trackComponentEngaged(push: Push, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, push.requestId, timestamp)
    }

    /**
     * Track a push component being engaged using a [DisplayablePush] object.
     *
     * @param push [DisplayablePush] associated with the component engaged event.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackComponentEngaged(push: DisplayablePush, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, push.parentId, timestamp)
    }

    /**
     * Track a push component being engaged using an requestId.
     *
     * @param pushRequestId The requestId of the [Push] being viewed.
     * @param timestamp Timestamp of the event.
     *
     */
    fun trackComponentEngaged(pushRequestId: String, timestamp: Long = System.currentTimeMillis()){
        track(Action.COMPONENT_ENGAGED, pushRequestId, timestamp)
    }
}