package com.flybits.android.push.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.flybits.android.push.PushScope;
import com.flybits.android.push.deserializations.DeserializePushTemplate;
import com.flybits.android.push.models.results.PushTemplateResult;
import com.flybits.android.push.utils.PushTemplateQueryParameters;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;

import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@link PushTemplate} class is used to define all the necessary parameters associated to a
 * push notification that is sent to user's based on changing contextual values and hence the
 * changing of rule evaluation states.
 */
public class PushTemplate {

    static final String API         = PushScope.ROOT + "/pushrequests";
    static final String USER_SCOPE  = "/userscoped";

    private String action;
    private Context context;
    private String id;
    private PushTemplateOptions options;
    private String message;
    private String name;
    private String title;
    private HashMap<String, String> customFields;
    private HashMap<String, String> listOfUsers;
    private HashMap<String, Integer> listOfRules;

    private PushTemplate(Builder builder){
        this.context        = builder.context;
        this.title          = builder.title;
        this.message        = builder.message;
        this.name           = builder.name;

        if (builder.id != null){
            id              = builder.id;
        }
        if (builder.action != null){
            action          = builder.action;
        }
        if (builder.customFields != null){
            customFields    = builder.customFields;
        }
        if (builder.options != null){
            options = builder.options;
        }
        if (builder.listOfRules != null && builder.listOfRules.size() > 0){
            listOfRules     = builder.listOfRules;
        }
        if (builder.listOfUsers != null && builder.listOfUsers.size() > 0){
            listOfUsers     = builder.listOfUsers;
        }
    }

    /**
     * Create this {@code PushTemplate} based on the attributes defined within this object.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code ObjectResult} that contains the network request for creating
     * {@code PushTemplate}.
     */
    public ObjectResult<PushTemplate> create(@NonNull final Context mContext, @NonNull final ObjectResultCallback<PushTemplate> callback){
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<PushTemplate> query = new ObjectResult<PushTemplate>(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {

                try {
                    DeserializePushTemplate deserializer    = new DeserializePushTemplate(context);
                    String jsonBody                         = deserializer.toJson(PushTemplate.this);
                    final Result<PushTemplate> result       = FlyAway.post(context, API + USER_SCOPE, jsonBody, deserializer, "PushTemplate.create", PushTemplate.class);
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setFailed(e);
                }
            }
        });
        return query;
    }

    /**
     * Deletes a specific {@link PushTemplate} from the list of templates created by the connected
     * user. This will remove any pending triggered notifications.
     *
     * @param mContext The state of the application.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public BasicResult delete(@NonNull final Context mContext,
                              @NonNull final BasicResultCallback callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult resultObject    = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                try {
                    final Result result = FlyAway.delete(context, API, "PushTemplate.delete", id + USER_SCOPE);
                    resultObject.setResult(result);
                }catch(final FlybitsException e){
                    resultObject.setFailed(e);
                }
            }
        });
        return resultObject;
    }

    /**
     * Retrieve all the {@code PushTemplate}s that have been created by the connected {@code User}.
     *
     * @param mContext The state of the application.
     * @param options The {@link PushTemplateQueryParameters} which allow you to easily define
     *                the pagination schema for the request you are trying to make.
     * @param callback The callback that initiated when the request is completed. It will contain
     *                 either a successful method or failure with a
     *                 {@code FlybitsException} which indicates the reason for failure.
     * @return The {@link ExecutorService} which can be shutdown by the application developer in
     * case the request is taking too long or the user no longer needs the result from the request.
     */
    public static PushTemplateResult get(@NonNull final Context mContext, @NonNull final PushTemplateQueryParameters options,
                                 final PagedResultCallback<PushTemplate> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final PushTemplateResult query = new PushTemplateResult(mContext, callback, executorService, options, handler);

        executorService.execute(new Runnable() {
            public void run() {
                try {
                    DeserializePushTemplate singleDersializer   = new DeserializePushTemplate(mContext);
                    final Result<PagedResponse<PushTemplate>> result  = FlyAway.get(mContext, API+USER_SCOPE, options, new DeserializePagedResponse<PushTemplate>(singleDersializer), "PushTemplate.get");
                    query.setResult(result);
                } catch (final FlybitsException e) {
                    query.setResult(new Result<PagedResponse<PushTemplate>>(e, "Push.get() failed"));
                }
            }
        });
        return query;
    }

    /**
     * Get the action associated to the {@code PushTemplate}.
     *
     * @return The action that corresponds to the {@code PushTemplate}.
     */
    public String getAction() {
        return action;
    }

    /**
     * Get the list of custom fields that are associated to the {@code PushTemplate}.
     *
     * @return A HashMap of the keys and values associated to JSON keys and values that should be
     * added to the payload of the {@code PushTemplate}. If no keys and values are set then an empty
     * HashMap will be returned.
     */
    public HashMap<String, String> getCustomFields() {
        if (customFields    == null){
            return new HashMap<>();
        }
        return customFields;
    }

    /**
     * Get the unique identifier set by the server when this {@code PushTemplate} is/was created.
     *
     * @return The String representation of the unique identifier.
     */
    public String getId(){
        return id;
    }

    /**
     * Get the list of rules and their results that are associated to the {@code PushTemplate}.
     *
     * @return A HashMap of the rules with their evaluation results that should be added to the
     * {@code PushTemplate}. If no keys and values are set then an empty HashMap will be returned.
     */
    public HashMap<String, Integer> getListOfRules() {
        if (listOfRules    == null){
            return new HashMap<>();
        }
        return listOfRules;
    }

    /**
     * Get the list of users identifiers along with those user's device identifier that this
     * {@code PushTemplate} will send push notifications to once a rule has triggered such an event.
     *
     * @return A HashMap of the user identifiers along with their device identifiers that should be
     * added to the {@code PushTemplate}. If no keys and values are set then an empty HashMap will
     * be returned.
     */
    public HashMap<String, String> getListOfUsers() {
        if (listOfUsers    == null){
            return new HashMap<>();
        }
        return listOfUsers;
    }

    /**
     * Get the optional {@link PushTemplateOptions} that are associated to the {@code PushTemplate}.
     * Through these options applications will be able to set throttling options that indicates how
     * often push notifications should be sent to the user.
     *
     * @return {@link PushTemplateOptions} that should be associated to the {@code PushTemplate} in
     * order to indicate any throttling options associated to the push notification.
     */
    public PushTemplateOptions getOptions() {
        return options;
    }

    /**
     * Get the message text that is associated to the {@code PushTemplate}.
     *
     * @return The message text that should be displayed to the user.
     */
    public String getMessage() {
        return message;
    }

    /**
     * Get the name of given to the {@code PushTemplate} so that is can be quickly found.
     *
     * @return The name of the {@code PushTemplate}.
     */
    public String getName() {
        return name;
    }

    /**
     * Get the title text that is associated to the {@code PushTemplate}.
     *
     * @return The title text that should be displayed to the user.
     */
    public String getTitle() {
        return title;
    }


    /**
     * The {@code Builder} class that defines all the possible filter options that can be used when
     * constructing a FCM Push notification.
     */
    public static class Builder {

        private Context context;
        private String title;
        private String message;
        private String action;
        private String name;
        private String id;
        private PushTemplateOptions options;
        private HashMap<String, String> customFields;
        private HashMap<String, String> listOfUsers;
        private HashMap<String, Integer> listOfRules;

        /**
         * Default Constructor that defines an empty {@code Builder}.
         *
         * @param context The application context.
         * @param name The name of the Push Template to be created.
         * @param title The title or heading of the Push Notification to be sent.
         * @param message The message of the Push Notification to be sent.
         */
        public Builder(Context context, String name, String title, String message){
            this.context        = context;
            this.title          = title;
            this.message        = message;
            this.action         = "custom";
            this.name           = name;
            this.listOfRules    = new HashMap<>();
            this.listOfUsers    = new HashMap<>();
            this.customFields   = new HashMap<>();
        }

        /**
         * Adds a custom field item to the payload of the Push Notification.
         *
         * @param key The {@code key} of the Custom Field item.
         * @param value The {@code value} of the Custom Field time based on the {@code key}.
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder addCustomField(@NonNull String key, @NonNull String value){
            this.customFields.put(key, value);
            return this;
        }

        /**
         * Adds a {@code Rule} the push notification which indicates that the push notification is
         * triggered by {@code Rule} changes.
         *
         * @param ruleID The unique identifier of the {@code Rule}.
         * @param result The result which indicates whether or not the {@code Rule} should trigger a
         *               push notification when a {@code Rule} evaluation changes from false to
         *               true (1), or from true to false (2) or just when the result changes (3).
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder addRule(@NonNull String ruleID, ResultType result) throws IllegalArgumentException{
            if (result == ResultType.UNKNOWN){
                throw new IllegalArgumentException("The result cannot be ResultType.UNKNOWN");
            }
            listOfRules.put(ruleID, result.getKey());
            return this;
        }

        /**
         * Adds a {@code User} the push notification which indicates that the push notification
         * should be sent to the a specific user/device combination.
         *
         * @param userID The unique identifier of the {@code User}.
         * @param deviceID The unique identifier of {@code User}'s {@code device}. This is important
         *                 as you should only target a specific device not an entire user in the
         *                 event that they have registered multiple devices.
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder addUser(@NonNull String userID, @NonNull String deviceID) throws IllegalArgumentException{
            listOfUsers.put(userID, deviceID);
            return this;
        }

        /**
         * Sets the action attribute associated to the push notification.
         *
         * @param action The action associated to this push notification.
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder setAction(String action){
            this.action     = action;
            return this;
        }

        /**
         * Sets a unique identifier that is obtained from the server when retrieving a list of
         * {@code PushTemplate}s. If a user is manually create a {@code PushTemplate} then this
         * field does not need to called.
         *
         * @param id The unique identifier of the {@code PushTemplate}.
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder setID(@NonNull String id){
            this.id = id;
            return this;
        }

        /**
         * Set additional options that should be included as part of the push notification. These
         * options will allow you to throttle the amount of notifications that are set.
         *
         * @param max The maximum number of push notifications that should be sent for this
         *            instance. If there is no {@code max} then -1 should be used.
         * @param delay Once a rule changes from true to false or from false to true, there should
         *              be a delay of {@code delay} seconds before the nothing is sent. If there is
         *              no {@code delay} then -1 should be used.
         * @param cooldown The time in seconds that should be waited after the {@code max} is
         *                 reached but resetting the {@code max} value to zero. If no
         *                 {@code cooldown} should take place then -1 should be used.
         * @return A {@code Builder} object where additional attributes can be set.
         */
        public Builder setOptions(int max, int delay, int cooldown){
            this.options    = new PushTemplateOptions(max, delay, cooldown);
            return this;
        }

        /**
         * Build the {@code PushTemplate_AsBuilder} object to be used for constructing a push
         * notification.
         *
         * @return {@code PushTemplate_AsBuilder} object containing information about the push
         * notification.
         */
        public PushTemplate build() throws IllegalArgumentException {

            if (listOfRules == null || listOfRules.size() == 0) {
                throw new IllegalArgumentException("A valid Push Notification must have called the addRule(String, int) method called at least once.");
            }
            return new PushTemplate(this);
        }
    }

    /**
     * The Result associated to the {@code Rule}. A result can be either a rule evaluation being
     * changed from true to false, false to true, or both.
     */
    public enum ResultType{

        /**
         * Indicates that the Result should trigger from false to true.
         */
        CHANGED_TO_TRUE(1),

        /**
         * Indicates that the Result should trigger from true to false.
         */
        CHANGED_TO_FALSE(2),

        /**
         * Indicates that the Result can be change from either true to false, or false to true.
         */
        CHANGED_TO_TRUE_OR_FALSE(3),

        /**
         * An unknown result was selected.
         */
        UNKNOWN(-99);

        private final int key;

        /**
         * Constructor that defines the key for each {@code ResultType} option.
         *
         * @param key the integer value representing each {@code ResultType} option.
         */
        ResultType(int key) {
            this.key = key;
        }

        /**
         * Get the String representation for the {@code ResultType} option.
         *
         * @return String representation of the {@code ResultType} option.
         */
        public int getKey() {
            return this.key;
        }

        /**
         * Get the {@code ResultType} enum value corresponding to an Integer representation.
         *
         * @param key the Integer representation of the {@code ResultType} enum.
         *
         * @return The {@code ResultType} enum for the Integer representation.
         */
        public static ResultType fromKey(int key) {
            for(ResultType type : ResultType.values()) {
                if(type.getKey() == key) {
                    return type;
                }
            }
            return UNKNOWN;
        }

    }
}
