package com.flybits.android.kernel.api

import android.content.Context
import com.flybits.android.kernel.deserializers.DeserializeContentDatum
import com.flybits.android.kernel.models.Content
import com.flybits.android.kernel.models.PagedArray
import com.flybits.android.kernel.models.SurveyMetadata
import com.flybits.android.kernel.models.internal.ContentDataResponse
import com.flybits.android.kernel.utilities.ContentDataParameters
import com.flybits.commons.library.exceptions.NetworkResponseException
import com.flybits.commons.library.exceptions.FlybitsException
import com.flybits.commons.library.http.HttpDefaultClass
import com.flybits.commons.library.http.RequestStatus
import com.flybits.commons.library.logging.Logger
import com.flybits.commons.library.models.internal.Result

import java.io.IOException

object FlyContentData {

    @JvmField
    internal val KEY_DATA_TYPE_SURVEY = "SurveyQuestions"
    @JvmField
    internal val API_SUBMIT_SURVEY_DATA = "/kernel/content/survey/instances/%1\$s/data"
    @JvmField
    internal val API_CONTENT_DATA = "/kernel/content/instances/%1\$s/data"
    @JvmField
    internal val API_CONTENT_DATA_PAGED =
        "$API_CONTENT_DATA?fields=%2\$s&expand=%2\$s&%2\$s.offset=%3\$d&%2\$s.limit=%4\$d"

    /**
     * Retrieves a paged portion of data from a content instance. This is the API endpoint of [PagedArray].
     *
     * @param context The [Context] of the application.
     * @param array The [PagedArray] this request is being called for.
     * @param offset The paged offset to start from.
     * @param limit The number of list entries to request.
     * @param deserializer Used to deserialize the response, optional, will be generated by default.
     * @param <T> The POJO class this [PagedArray] contains.
     *
     * @return A [Result] object for setting callbacks and canceling the request.
     */
    @JvmStatic
    @JvmOverloads
    @Throws(FlybitsException::class)
    fun <T> getPaged(
        context: Context,
        array: PagedArray<*>,
        offset: Long,
        limit: Long,
        deserializer: DeserializeContentDatum<T> = DeserializeContentDatum(
            context,
            array.instanceId,
            array.templateModel
        )
    ): Result<ContentDataResponse<T>> {

        return try {
            HttpDefaultClass.Builder(
                context,
                true,
                String.format(
                    API_CONTENT_DATA_PAGED,
                    array.instanceId,
                    array.name,
                    offset,
                    limit
                )
            ).get().response.let { result ->
                if (result.status != RequestStatus.COMPLETED) {
                    result as Result<ContentDataResponse<T>>
                } else deserializer.fromJson(result.resultAsString)?.also { response ->
                    result.setResponse(response)
                }?.let {
                    result as Result<ContentDataResponse<T>>
                }?: Result(
                    FlybitsException("Parsing Error"),
                    result.resultAsString
                )
            }
        } catch (e: IOException) {
            Logger.exception("FlyContentData.get", e)
            throw NetworkResponseException()
        }

    }

    /**
     * Perform a GET request to the Content Data API, given a set of params and a POJO to deserialize the response into.
     *
     * @param context The [Context] of the application.
     * @param params Any params that define how the response will be paged, or a unique ID to retrieve.
     * @param classObj The POJO to deserialize the data into.
     * @param deserializer Used to deserialize the response, optional, will be generated by default.
     *
     * @return [Result] object is returned so you can set the response callback.
     *
     */
    @JvmStatic
    @JvmOverloads
    @Throws(FlybitsException::class)
    operator fun <T> get(
        context: Context,
        params: ContentDataParameters,
        classObj: Class<*>,
        deserializer: DeserializeContentDatum<T> = DeserializeContentDatum(
            context,
            params.instanceID,
            classObj
        )
    ): Result<ContentDataResponse<T>> {

        return try {
            HttpDefaultClass.Builder(
                context,
                true,
                String.format(
                    API_CONTENT_DATA,
                    params.instanceID
                )
            ).get().response.let { result ->
                if (result.status != RequestStatus.COMPLETED) {
                    result as Result<ContentDataResponse<T>>
                } else deserializer.fromJson(result.resultAsString)?.also { response ->
                    result.setResponse(response)
                }?.let {
                    result as Result<ContentDataResponse<T>>?
                }?: Result(
                    FlybitsException("Parsing Error"),
                    result.resultAsString
                )
            }
        } catch (e: IOException) {
            Logger.exception("FlyContentData.get", e)
            throw NetworkResponseException()
        }

    }

    /**
     * Perform a POST request to the Content Data API, to save any content data (IE: Survey).
     *
     * @param context The [Context] of the application.
     * @param content The [Content] this data belongs to.
     * @param json The data to save to the content data instance.
     *
     * @return [Result] object is returned so you can set the response callback.
     */
    @JvmStatic
    @Throws(FlybitsException::class)
    fun <T> post(context: Context, content: Content, json: String): Result<T> {
        // survey submission requires a specific URL
        return when (content.type) {
            KEY_DATA_TYPE_SURVEY -> {
                val surveyMetadata =
                    if (content.metadata is SurveyMetadata) content.metadata as SurveyMetadata else SurveyMetadata()
                String.format(API_SUBMIT_SURVEY_DATA, surveyMetadata.surveyAnswersInstanceID)
            }
            else -> String.format(API_CONTENT_DATA, content.id)
        }.let { url ->
            try {
                HttpDefaultClass.Builder(context, true, url)
                    .post(json)
                    .response as Result<T>
            }catch (e: IOException) {
                Logger.exception("FlyContentData.post", e)
                throw NetworkResponseException()
            }

        }
    }
}
