package com.flybits.android.kernel.db.dao;

import android.arch.lifecycle.LiveData;
import android.arch.paging.DataSource;
import android.arch.persistence.room.Dao;
import android.arch.persistence.room.Insert;
import android.arch.persistence.room.Query;
import android.arch.persistence.room.Transaction;
import android.content.Context;

import com.flybits.android.kernel.models.Content;
import com.flybits.internal.db.CommonsDatabase;

import java.util.List;

import static android.arch.persistence.room.OnConflictStrategy.REPLACE;

@Dao
//TODO: Missing Comments
public abstract class ContentDao {

    /**
     * Get Requests
     */
    @Query("SELECT * FROM content WHERE id = :id")
    public abstract Content getById(String id);

    @Query("SELECT * FROM content")
    public abstract List<Content> getAll();

    @Query("SELECT * FROM content WHERE id IN (:ids)")
    public abstract List<Content> getByIdList(List<String> ids);

    @Query("SELECT * FROM content WHERE id IN (:ids) LIMIT :limit")
    public abstract List<Content> getByIdList(List<String> ids, int limit);

    @Query("SELECT * FROM content WHERE id IN (:ids) LIMIT :limit OFFSET :offset")
    public abstract List<Content> getByIdList(List<String> ids, int limit, int offset);

    @Query("SELECT * FROM content WHERE id IN (:ids) ORDER BY sequence LIMIT :limit OFFSET :offset")
    public abstract List<Content> getByIdListOrdered(List<String> ids, int limit, int offset);

    @Query("SELECT * FROM content WHERE templateId = :templateId")
    public abstract List<Content> getAllByTemplateId(String templateId);

    @Query("SELECT * FROM content WHERE type = :type")
    public abstract List<Content> getByType(String type);

    @Query("SELECT * FROM content WHERE type IN (:typeList)")
    public abstract List<Content> getByTypeList(List<String> typeList);

    /**
     * BASIC CRUD
     */
    @Insert(onConflict = REPLACE)
    public abstract void insert(List<Content> content);

    @Query("DELETE FROM content")
    public abstract void clear();

    @Query("DELETE FROM content WHERE id IN (:ids)")
    public abstract void deleteByIds(List<String> ids);

    /**
     * LIVE DATA
     */
    @Query("SELECT * FROM content")
    public abstract LiveData<List<Content>> getLive();

    @Query("SELECT * FROM content WHERE id IN (:ids)")
    public abstract LiveData<List<Content>> getLiveByIdList(List<String> ids);

    @Query("SELECT * FROM content WHERE id = :id")
    public abstract LiveData<List<Content>> getLiveById(String id);

    @Query("SELECT * FROM content WHERE templateId = :templateId")
    public abstract LiveData<List<Content>> getLiveAllByTemplateId(String templateId);

    @Query("SELECT * FROM content WHERE type = :type")
    public abstract LiveData<List<Content>> getLiveByType(String type);

    @Query("SELECT * FROM content WHERE type IN (:typeList)")
    public abstract LiveData<List<Content>> getLiveByTypeList(List<String> typeList);

    /**
     * DATA SOURCE for LivePagedListBuilder
     */
    @Query("SELECT * FROM content")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContent();

    @Query("SELECT * FROM content WHERE id IN (:ids)")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContentByIdList(List<String> ids);

    @Query("SELECT * FROM content WHERE id = :id")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContentById(String id);

    @Query("SELECT * FROM content WHERE templateId = :templateId")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContentAllByTemplateId(String templateId);

    @Query("SELECT * FROM content WHERE type = :type")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContentByType(String type);

    @Query("SELECT * FROM content WHERE type IN (:typeList)")
    public abstract DataSource.Factory<Integer, Content> getDataSourceContentByTypeList(List<String> typeList);

    @Transaction
    public void insertAndDeleteInTransaction(List<String> deleteContentIds, List<Content> insertContent) {
        // Anything inside this method runs in a single transaction.
        deleteByIds(deleteContentIds);
        insert(insertContent);
    }

    @Transaction
    public LiveData<List<Content>> getCached(Context context, String cacheId) {
        List<String> cachedIds = CommonsDatabase.getDatabase(context).cachingEntryDAO().getIdsByCachingKey(cacheId);
        return getLiveByIdList(cachedIds);
    }

    @Transaction
    public List<Content> getByCacheId(Context context, String cacheId, int limit, int offset) {
        List<String> cachedIds = CommonsDatabase.getDatabase(context).cachingEntryDAO().getIdsByCachingKey(cacheId);
        return getByIdListOrdered(cachedIds, limit, offset);
    }
}
