package com.flybits.android.kernel.utilities;

import android.support.annotation.NonNull;

import com.flybits.android.kernel.models.Content;
import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;

import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link ContentParameters} class is used to define all the parameters set for the GET
 * request associated to {@link Content}.
 */
public class ContentParameters  extends QueryParameters {

    private String locId;
    private double latitude;
    private double longitude;
    private double radius;
    private String url;
    private String templateId;
    private String templateType;
    private String groupId;

    private ContentParameters(ContentParameters.Builder builder) {
        super(builder);

        this.templateId = builder.templateId;
        this.templateType = builder.templateType;
        this.locId = builder.locId;

        if (locId != null) {
            this.latitude = builder.latitude;
            this.longitude = builder.longitude;
            this.radius = builder.radius;
        }

        if (builder.experienceId != null){
            url = String.format(Content.API_CONTENT_INSTANCE_VIA_EXPERIENCE, builder.experienceId);
        } else if (locId != null || templateId != null){
            url = Content.API_CONTENT_INSTANCE;
        } else{
            url = Content.API_CONTENT_RELEVANT_INSTANCES;
            this.groupId = builder.groupId;
        }
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();
        if (locId != null){

            ArrayList<String> latitudeParam = new ArrayList<>();
            latitudeParam.add( String.valueOf(latitude));
            params.put("lat", latitudeParam);

            ArrayList<String> longitudeParam = new ArrayList<>();
            longitudeParam.add( String.valueOf(longitude));
            params.put("lng", longitudeParam);

            ArrayList<String> radiusParam = new ArrayList<>();
            radiusParam.add( String.valueOf(radius));
            params.put("radius", radiusParam);

            ArrayList<String> locParam = new ArrayList<>();
            locParam.add( String.valueOf(locId));
            params.put("loc", locParam);
        }

        if (templateId != null) {
            ArrayList<String> templateIdParam = new ArrayList<>();
            templateIdParam.add(String.valueOf(templateId));
            params.put("templateId", templateIdParam);
        }

        if (templateType != null) {
            ArrayList<String> templateTypeParam = new ArrayList<>();
            templateTypeParam.add(String.valueOf(templateType));
            params.put("templateType", templateTypeParam);
        }

        if (groupId != null){
            ArrayList<String> groupIdParam = new ArrayList<>();
            groupIdParam.add(groupId);
            params.put("groupId", groupIdParam);
        }

        ArrayList<String> dataParam = new ArrayList<>();
        dataParam.add( String.valueOf(true));
        params.put("data", dataParam);
        return params;
    }

    /**
     * Get the URL of the API that should be used to retrieve {@link Content}.
     *
     * @return The URL of the API that should be used to retrieve {@link Content}.
     */
    public String getUrl() {
        return url;
    }

    /**
     * {@code Builder} class used the create an {@link ContentParameters} class, which defines
     * all parameters used within the GET request for
     * {@link com.flybits.android.kernel.models.Content}.
     */
    public static class Builder extends QueryBuilder<Builder>{

        private String locId;
        private double latitude;
        private double longitude;
        private double radius;
        private String templateId;
        private String templateType;
        private String experienceId;
        private String groupId;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            super();
        }

        /**
         * Set the caching key for the query. This is useful because it allows you to later retrieve
         * cached copies of the query that you just made.
         *
         * @param cachingKey A unique key that represents the query. If no {@code cachingKey} is set a
         *                   hash of the query parameters will be used.
         * @param limit The maximum number of entities that can be cached.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setCaching(@NonNull String cachingKey, int limit){
            super.setCaching(cachingKey, limit);
            return this;
        }

        /**
         * Sets the experience Id for which Content should be retrieved for.
         *
         * @param id The identifier that represents an {@link com.flybits.android.kernel.models.Experience}.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setExperienceId(@NonNull String id){
            this.experienceId = id;
            return this;
        }

        /**
         * Set the group Id for which Content should be retrieved for. Only Content that is found in
         * {@link com.flybits.android.kernel.models.Group} based on the {@code id} will be
         * displayed.
         * @param id The identifier that represents an {@link com.flybits.android.kernel.models.Group}.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setGroupId(@NonNull String id){
            this.groupId = id;
            return this;
        }

        /**
         * Set sorting based on location. Therefore, the content will be sorted based on the
         * location data type.
         *
         * @param loc  location field to search on (required).
         * @param latitude latitude to search from (required).
         * @param longitude longitude to search from (required).
         * @param radius radius to search in.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setLocation(@NonNull String loc, double latitude, double longitude, long radius) {
            return setLocation(loc, latitude, longitude, radius, null);
        }

        /**
         * Set sorting based on location. Therefore, the content will be sorted based on the
         * location data type.
         *
         * @param loc  location field to search on (required).
         * @param latitude latitude to search from (required).
         * @param longitude longitude to search from (required).
         * @param radius radius to search in.
         * @param templateId The unique identifier of the Content Template to perform the search
         *                   for.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setLocation(@NonNull String loc, double latitude, double longitude,
                                   long radius, String templateId) {
            this.locId      = loc;
            this.latitude   = latitude;
            this.longitude  = longitude;
            this.radius     = radius;
            this.templateId = templateId;
            return this;
        }

        /**
         * Set the unique template identifier of the content that should be retrieved. Therefore,
         * only content of type {@code id} will be retrieved.
         *
         * @param id The unique template identifier for the {@link Content} you would like to
         *           retrieve.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         * @deprecated Use {@link #setTemplateType(String)}.
         */
        @Deprecated
        public Builder setTemplateId(@NonNull String id){
            this.templateId = id;
            return this;
        }

        /**
         * Set the template type of the {@link Content} that should be returned. This can be
         * retrieve a Content\'s {@link Content#getType()} method.
         *
         * @param type The type identifier for the {@link Content} you would like to retrieve.
         * @return The {@link Builder} that can be used to add other Content parameters for.
         */
        public Builder setTemplateType(@NonNull String type){
            this.templateType = type;
            return this;
        }

        /**
         * Create a {@link ContentParameters} object that contains all parameters associated to
         * the GET request for {@link com.flybits.android.kernel.models.Experience}.
         *
         * @return {@link ContentParameters} which is used to create a valid
         * {@link com.flybits.android.kernel.models.Experience} GET request.
         *
         * @throws RuntimeException when setExperienceId(...) and setLocation(...) are both called
         * as part of {@link ContentParameters} creation using the this method.
         */
        public ContentParameters build() throws RuntimeException{

            if (locId != null && experienceId != null){
                throw new RuntimeException("Currently joining the setExperienceId(...) and setLocation(...)" +
                        " methods is not supported. You must choose one or the other to build a" +
                        " valid ContentParameters object");
            }

            if (locId != null && groupId!= null){
                throw new RuntimeException("Currently joining the setGroupId(...) and setLocation(...)" +
                        " methods is not supported. You must choose one or the other to build a" +
                        " valid ContentParameters object");
            }

            if (experienceId != null && groupId!= null){
                throw new RuntimeException("Currently joining the setGroupId(...) and setExperienceId(...)" +
                        " methods is not supported. You must choose one or the other to build a" +
                        " valid ContentParameters object");
            }

            return new ContentParameters(this);
        }
    }
}
