package com.flybits.android.kernel.db.caching;

import android.app.Activity;
import android.content.Context;
import android.support.annotation.NonNull;

import com.flybits.android.kernel.db.KernelDatabase;
import com.flybits.android.kernel.models.Content;
import com.flybits.commons.library.caching.FlybitsCacheLoader.CacheListLoader;
import com.flybits.commons.library.caching.FlybitsCacheLoader;

import java.util.List;

/**
 * The {@code ContentCacheLoader} is responsible for loading cached data associated to a
 * {@link Content}. This data is inserted into the local SQLite DB whenever {@link Content} data is
 * received from the Flybits server.
 */
public class ContentCacheLoader extends FlybitsCacheLoader<Content> {

    /**
     * Constructor used to define a {@link ContentCacheLoader} class with the Activity parameter.
     *
     * @param activity The activity that created this object.
     */
    public ContentCacheLoader(Activity activity){
        super(activity);
    }

    @Override
    protected List<Content> load(List<String> cachedIds, int limit) {
        return KernelDatabase.getDatabase(getContext()).contentDao().getByCachedIds(cachedIds, limit);
    }

    @Override
    protected Content load(String id){
        return KernelDatabase.getDatabase(getContext()).contentDao().getById(id);
    }

    /**
     * This get method is used to retrieve the cached entities associated to the {@link Content}
     * model.
     *
     * @param cachingKey The caching key used to define which {@link Content} entities should be
     *                   retrieved.
     * @param callback The {@code CacheListLoader} callback that loads the cached data.
     * @return This {@link #ContentCacheLoader} object.
     */
    public ContentCacheLoader get(@NonNull String cachingKey, CacheListLoader<Content> callback){
        super.getList(cachingKey, callback);
        return this;
    }

    /**
     * This get method is used to retrieve the cached entities associated to the {@link Content}
     * model. This method contains a limit of the number of {@link Content} that should be
     * retrieved.
     *
     * @param cachingKey The caching key used to define which {@link Content} entities should be
     *                   retrieved.
     * @param limit The number entries that should be returned.
     * @param callback The {@code CacheListLoader} callback that loads the cached data.
     * @return This {@link #ContentCacheLoader} object.
     */
    public ContentCacheLoader get(@NonNull String cachingKey, int limit, CacheListLoader<Content> callback){
        super.getList(cachingKey, limit, callback);
        return this;
    }
}
