package com.flybits.context.utils;

import com.flybits.commons.library.models.internal.QueryBuilder;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.context.models.RuleScope;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.Map;

/**
 * The {@link RulesQueryParameters} class is used to define all the parameters set for the GET
 * request associated to {@link com.flybits.context.models.Rule}.
 */
public class RulesQueryParameters extends QueryParameters {

    private String scope;

    private RulesQueryParameters(Builder builder) {
        super(builder);
        if (builder.scope != null){
            this.scope      = builder.scope;
        }
    }

    /**
     * Get the list of query parameters that should be added to the GET request.
     *
     * @return A list of parameters that should be added to the GET query.
     */
    public Map<String, ArrayList<String>> getQueryParams(){

        Map<String, ArrayList<String>> params = super.getQueryParams();

        if (scope != null) {
            ArrayList<String>    listOfScopes   = new ArrayList<>();
            listOfScopes.add(scope);
            params.put("scope", listOfScopes);
        }
        return params;
    }

    /**
     * {@code Builder} class used the create an {@link RulesQueryParameters} class, which defines
     * all parameters used within the GET request for {@link com.flybits.context.models.Rule}.
     */
    public static class Builder extends QueryBuilder<Builder> {

        private String scope;

        /**
         * Default constructor to initializes all variables.
         */
        public Builder(){
            super();
        }

        /**
         * Set the scope which indicates whether a {@link com.flybits.context.models.Rule} should be
         * processed by everyone in the {@code Project} or only the current logged in {@code User}.
         * In most cases, when an administrator creates a {@link com.flybits.context.models.Rule} it
         * will have a {@link RuleScope#PROJECT} scope. However, whenever a {@code User} creates a
         * {@link com.flybits.context.models.Rule} for him/herself then the
         * {@link com.flybits.context.models.Rule} will have {@link RuleScope#USER} scope.
         *
         * @param scope The {@link RuleScope} that indicates {@link com.flybits.context.models.Rule}s
         *              should be retrieve that are evaluated for either the logged in {@code User}
         *              or everyone in the {@code Project}.
         *
         * @return A {@code Builder} object where additional {@code Options} filter settings can be
         * set.
         */
        public Builder setScope(RuleScope scope) throws InvalidParameterException{
            if (scope == RuleScope.UNKNOWN){
                throw new InvalidParameterException("Your Scope must be either User or Project");
            }
            this.scope  = scope.getKey();
            return this;
        }

        /**
         * Create a {@link RulesQueryParameters} object that contains all parameters associated to
         * the GET request for {@link com.flybits.context.models.Rule}.
         *
         * @return {@link RulesQueryParameters} which is used to create a valid
         * {@link com.flybits.context.models.Rule} GET request.
         */
        public RulesQueryParameters build(){
            return new RulesQueryParameters(this);
        }
    }
}
