package com.flybits.context.plugins.activity

import android.app.PendingIntent
import android.content.Context
import android.content.Intent
import android.support.v4.app.JobIntentService
import com.flybits.commons.library.logging.Logger
import com.google.android.gms.location.ActivityRecognitionClient
import com.google.android.gms.tasks.OnSuccessListener

/**
 * The [JobIntentService] that is initiated by a pending request obtained whenever Activity
 * Recognition process has begun. This {@code JobIntentService} is launched once the pending request is
 * completed and the result of the action has be obtained.
 */
private const val DETECTION_INTERVAL_IN_MILLISECONDS = 30 * 1000L;
private const val ACTIVITY_EVENT_JOB_ID = 5

class ActivityRecognitionJobIntent : JobIntentService() {
    override fun onCreate() {
        super.onCreate()
        if (instance == null) {
            instance = this
        }
    }

    override fun onHandleWork(intent: Intent) {
        requestActivityUpdatesButtonHandler()
    }

    private fun requestActivityUpdatesButtonHandler() {
        val task = mActivityRecognitionClient.requestActivityUpdates(
            DETECTION_INTERVAL_IN_MILLISECONDS,
            mPendingIntent
        )
        task.addOnSuccessListener(OnSuccessListener {
            Logger.appendTag(TAG).d("Successfully requested activity updates")
        })
        task.addOnFailureListener {
            Logger.appendTag(TAG).d("Requesting activity updates failed to start")
        }
    }


    companion object {
        private const val TAG = "ActivityRecognitionJobIntent"
        private var instance: JobIntentService? = null

        private val mActivityRecognitionClient: ActivityRecognitionClient by lazy {
            ActivityRecognitionClient(instance!!.applicationContext)
        }

        private val mPendingIntent: PendingIntent by lazy {
            PendingIntent.getBroadcast(
                instance, 1, mIntentService,
                PendingIntent.FLAG_UPDATE_CURRENT
            )
        }

        private val mIntentService by lazy {
            Intent(instance, ActivityRecognitionBroadcast::class.java)
        }


        fun enqueueWork(context: Context, intent: Intent) {
            enqueueWork(
                context,
                ActivityRecognitionJobIntent::class.java,
                ACTIVITY_EVENT_JOB_ID,
                intent
            )
        }

        fun removeActivityUpdatesButtonHandler() {
            if (instance != null) {
                val task = mActivityRecognitionClient.removeActivityUpdates(
                    mPendingIntent
                )
                task.addOnSuccessListener {
                    Logger.appendTag(TAG).d("Removed activity updates successfully!")
                }
                task.addOnFailureListener {
                    Logger.appendTag(TAG).d("Failed to remove activity updates!")
                }
            } else Logger.appendTag(TAG).d("No Activity plugin registered ")
        }
    }
}