package com.flybits.context.plugins.activity

import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import com.flybits.commons.library.logging.Logger.appendTag
import com.flybits.context.plugins.activity.ActivityData.ActivityType
import com.google.android.gms.location.ActivityRecognitionResult
import com.google.android.gms.location.DetectedActivity

/***
 * The [ActivityRecognitionBroadcast] is a [BroadcastReceiver] that response to map the Activity Recognition [DetectedActivity] to [ActivityData]
 * The  [ActivityData] then will flush to the Flybits server via API: context/ctxrepo
 */
class ActivityRecognitionBroadcast : BroadcastReceiver() {
    private val tagDevice = this.javaClass.simpleName

    override fun onReceive(context: Context, intent: Intent) {
        appendTag(tagDevice).d("Fetching Data ... ")

        val data = processIntent(intent)
        data?.let {
            appendTag(tagDevice).d(
                "Received Data: $data"
            )
            val timeInSec = if (data.time > 0) data.time else System.currentTimeMillis() / 1000
            data.update(context, timeInSec)
        }
    }

    private fun processIntent(intent: Intent): ActivityData? {
        if (ActivityRecognitionResult.hasResult(intent)) {
            val result = ActivityRecognitionResult.extractResult(intent)
            var driving = false
            var cycling = false
            var stationary = false
            var running = false
            var walking = false
            var unknown = false
            val detectedActivity = result.probableActivities
            for (activity in detectedActivity) {
                if (activity.confidence > 10) {
                    when (getActivityName(activity.type)) {
                        ActivityType.IN_VEHICLE -> driving = true
                        ActivityType.ON_BICYCLE -> cycling = true
                        ActivityType.STILL -> stationary = true
                        ActivityType.RUNNING -> running = true
                        ActivityType.WALKING -> walking = true
                        else -> unknown = true
                    }
                }
            }
            return ActivityData(cycling, driving, running, stationary, walking, unknown)
        } else {
            appendTag(tagDevice).e("Intent had no data returned")
        }
        return null
    }

    private fun getActivityName(type: Int): ActivityType {
        return when (type) {
            DetectedActivity.IN_VEHICLE -> ActivityType.IN_VEHICLE
            DetectedActivity.ON_BICYCLE -> ActivityType.ON_BICYCLE
            DetectedActivity.ON_FOOT -> ActivityType.WALKING
            DetectedActivity.WALKING -> ActivityType.WALKING
            DetectedActivity.STILL -> ActivityType.STILL
            DetectedActivity.TILTING -> ActivityType.STILL
            DetectedActivity.RUNNING -> ActivityType.RUNNING
            DetectedActivity.UNKNOWN -> ActivityType.UNKNOWN
            else -> ActivityType.UNKNOWN
        }
    }
}