package com.flybits.context.plugins.fitness

import android.content.Context
import android.os.Bundle
import androidx.work.WorkerParameters
import com.flybits.commons.library.logging.Logger
import com.flybits.context.models.ContextData
import com.flybits.context.services.FlybitsContextPluginsWorker
import java.util.concurrent.TimeUnit

const val TAG = "PluginFitness"

/**
 * The [FitnessContextPluginWorker] is a worker responsible for retrieving the Fitness data.
 * This will be triggered by `WorkManager` on start of fitness plugin to fetch the fitness information of the authenticated account
 * after specified interval.
 * The minimum interval for the worker to execute is 900 seconds.
 * Permissions required are : FitnessOptions (tobe explicitly asked).
 */
class FitnessContextPluginWorker(context: Context, workerParameters: WorkerParameters) :
    FlybitsContextPluginsWorker(context, workerParameters), IFitnessData {

    var totalSteps: Long = 0
    override fun onGetSteps(steps: Long) {
        totalSteps = steps
    }

    override fun onCancel() {
        Logger.appendTag(TAG).d("Error Fetching Fitness Data.")
    }

    override fun getData(): ContextData {
        Logger.appendTag(TAG).d("Fetching Fitness Data ...")
        try {
            val fitnessUtils = FitnessUtils(applicationContext)
            fitnessUtils.getStepsToday(this)
            wait(5, TimeUnit.SECONDS)
            return FitnessData(totalSteps)
        } catch (e: Exception) {
            throw Exception(e.message)
        }
    }

    override fun getRequiredPermissions(): Array<String> {
        return arrayOf()
    }

    override fun initialize(bundle: Bundle?) {
    }

    override fun isSupported(): Boolean {
        return true
    }

    private fun wait(time: Long, unit: TimeUnit) {
        val mServiceStartTime = System.currentTimeMillis()
        val waitInMilli = unit.toMillis(time)

        while (System.currentTimeMillis() - mServiceStartTime <= waitInMilli) {
            Thread.yield()
        }
    }
}