package com.flybits.context.plugins.oauth;

import android.os.Parcel;
import android.os.Parcelable;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code AccessToken} class is used to define a unique token that is obtained from a 3rd-party
 * OAuth service. This token will allow you get information about the user that can be used for
 * retrieving Contextual data about the User.
 */
public class AccessToken  implements Parcelable{

    private String provider;
    private String accessToken;
    private Long expiresAt;

    /**
     * Constructor used for un-flattening a {@code AccessToken} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code AccessToken} parcel.
     */
    protected AccessToken(Parcel in){
        provider     = in.readString();
        accessToken  = in.readString();
        expiresAt    = in.readLong();
    }

    /**
     * Constructor used to define an {@code AccessToken} received from a 3rd-party OAuth source.
     *
     * @param provider The 3rd-party OAuth provider from where the access token is retrieved from.
     * @param accessToken The unique token associated to the User after they have logged in.
     * @param expiresAt Indicates in epoch time when this {@code AccessToken} expires.
     */
    public AccessToken(String provider, String accessToken, Long expiresAt) {
        this.provider     = provider;
        this.accessToken  = accessToken;
        this.expiresAt    = expiresAt;
    }

    /**
     * Get the unique access token associated to the 3rd-party OAuth user that has logged in.
     *
     * @return The String representation of the unique token received from the 3rd-party OAuth
     * service.
     */
    public String getAccessToken() {
        return accessToken;
    }

    /**
     * Get the epoch time that indicates when the {@link #getAccessToken()} expires.
     *
     * @return The epoch time that indicates when the {@link #getAccessToken()} expires.
     */
    public Long getExpiresAt() {
        return expiresAt;
    }

    /**
     * Get the 3rd-party OAuth service provider from where the {@link #getAccessToken()} is
     * obtained.
     *
     * @return The String representation of where the {@link #getAccessToken()} is obtained from.
     */
    public String getProvider() {
        return provider;
    }


    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(provider);
        out.writeString(accessToken);
        out.writeLong(expiresAt);
    }

    public void fromJson(String json) {

        try {
            JSONObject jsonObj  = new JSONObject(json);
            if (!jsonObj.isNull("accessToken")) {
                accessToken = jsonObj.getString("accessToken");
            }
            if (!jsonObj.isNull("provider")) {
                provider = jsonObj.getString("provider");
            }
            if (!jsonObj.isNull("expiresAt")) {
                expiresAt = jsonObj.getLong("expiresAt");
            }
        }catch (JSONException exception){}

    }

    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            if (provider != null) {
                object.put("provider", provider);
            }
        }catch (JSONException exception){}

        try{
            if (accessToken != null) {
                object.put("accessToken", accessToken);
            }
        }catch (JSONException exception){}

        try{
            if (expiresAt != null) {
                object.put("expiresAt", expiresAt);
            }
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public String toString() {
        return toJson();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        AccessToken that = (AccessToken) o;

        if (provider != null ? !provider.equals(that.provider) : that.provider != null)
            return false;
        if (accessToken != null ? !accessToken.equals(that.accessToken) : that.accessToken != null)
            return false;
        return expiresAt != null ? expiresAt.equals(that.expiresAt) : that.expiresAt == null;
    }

    /**
     * Parcelable.Creator that instantiates {@code AccessToken} objects
     */
    public static final Creator<AccessToken> CREATOR = new Creator<AccessToken>(){
        public AccessToken createFromParcel(Parcel in){
            return new AccessToken(in);
        }

        public AccessToken[] newArray(int size){
            return new AccessToken[size];
        }
    };


}
