package com.flybits.context.utils;

import androidx.annotation.Nullable;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Calendar;

/**
 * The {@link RuleScheduler} class allows the application to define a repeatable schedule that
 * indicates when a {@link com.flybits.context.models.Rule} should be evaluated. For times that the
 * schedule is not active, the rule will not be processed as either true or false regardless of what
 * is sent to the server.
 */
public class RuleScheduler  {

    private long startTime, endTime = -1;
    private RepeatSchedule repeat;
    private String contextualStart = null;

    /**
     * Default constructor used to define the start time of the {@link com.flybits.context.models.Rule}
     * processing as well as the {@link RepeatSchedule} object which defines how often the
     * {@link com.flybits.context.models.Rule} processing is repeated.
     *
     * @param calendar The {@code Calendar} object that represents the time that the
     *                 {@link com.flybits.context.models.Rule} should be activated to allow for
     *                 Context changes.
     * @param repeat The {@link RepeatSchedule} object that contains information that indicates
     *               when the {@link com.flybits.context.models.Rule}'s evaluation is repeated.
     */
    public RuleScheduler(Calendar calendar, RepeatSchedule repeat){
        this.startTime  = calendar.getTimeInMillis() / 1000;
        this.repeat     = repeat;
    }

    /**
     * Constructor used to initialize a {@code RuleScheduler} with a start time for when the
     * {@link com.flybits.context.models.Rule} should be active.
     *
     * @param calendar The {@code Calendar} object that represents the time that the
     *                 {@link com.flybits.context.models.Rule} should be activated to allow for
     *                 Context changes.
     */
    public RuleScheduler(Calendar calendar){
        this(calendar,  new RepeatSchedule(RepeatSchedule.FrequencyType.ONCE, 1, 1));
    }

    /**
     * Indicates the end time of the {@link com.flybits.context.models.Rule} in epoch. After this
     * time is passed the {@link com.flybits.context.models.Rule} will not longer be processed.
     *
     * @param calendar The {@code Calendar} object that represents the time that the
     *                 {@link com.flybits.context.models.Rule} should be deactivated the processing
     *                 of Context changes.
     */
    public void setEndTime(Calendar calendar){
        this.endTime    = calendar.getTimeInMillis() / 1000;
    }

    /**
     * Convert to {@code RuleScheduler} into its JSON format so that it can be sent to the server as
     * part of the {@link com.flybits.context.models.Rule} creation process.
     *
     * @return The JSON representation of this {@code RuleScheduler} object.
     */
    @Nullable public JSONObject toJSONObject() {
        try {
            JSONObject object       = new JSONObject();

            JSONObject startObject  = new JSONObject();
            startObject.put("start", startTime);
            startObject.put("repeat", repeat.toJSONObject());

            if (contextualStart != null){
                startObject.put("contextualStart", contextualStart);
            }

            object.put("startScheduler", startObject);

            if (endTime > -1){
                JSONObject endObject  = new JSONObject();
                endObject.put("start", endTime);
                endObject.put("repeat", repeat.toJSONObject());
                object.put("endScheduler", endObject);
            }

            return object;
        }catch (JSONException e){
            return null;
        }
    }

    @Override
    @Nullable public String toString(){
        return toJSONObject().toString();
    }

    /**
     * This class represents how often the {@link RuleScheduler} should be repeated for processing.
     * For example, using this class we can indicate either a rule is processed on a weekly, monthly,
     * yearly basis and what the maximum number of times the {@link com.flybits.context.models.Rule}
     * should take place.
     */
    public static class RepeatSchedule{

        private long frequency;
        private long limit;
        private FrequencyType type;

        /**
         * The constructor used to define the minimum attributes needed to create a repeating
         * schedule for {@link com.flybits.context.models.Rule} processing.
         *
         * @param type Indicates the unit of repeat for the schedule such as daily, weekly, monthly,
         *             etc.
         * @param frequency In combination with {@code type}, it determines how often the schedule
         *                  repeats.
         * @param limit Determines the maximum number of times that schedule can repeat.
         */
        public RepeatSchedule(FrequencyType type, long frequency, long limit){
            if (type != null) {
                this.type       = type;
                this.frequency  = frequency;
                this.limit      = limit;
            }else{
                this.type       = FrequencyType.ONCE;
                this.frequency  = 0;
                this.limit      = -1;
            }
        }


        /**
         * Convert to {@code RepeatSchedule} into its JSON format so that it can be sent as part of
         * the {@link RuleScheduler} object.
         *
         * @return The JSON representation of this {@code RepeatSchedule} object.
         */
        @Nullable public JSONObject toJSONObject(){
            try {
                JSONObject object = new JSONObject();
                object.put("frequencyType", type.getKey());
                object.put("frequency", frequency);
                object.put("limit", limit);

                return object;
            }catch (JSONException e){
                return null;
            }
        }

        /**
         * The {@code FrequencyType} indicates the unit of repeat for the schedule such as daily,
         * weekly, monthly, etc.
         */
        public enum FrequencyType {

            ONCE("once"),

            HOURLY("hour"),

            DAILY("day"),

            WEEKLY("week"),

            MONTHLY("month"),

            YEARLY("year");

            private final String key;

            /**
             * Constructor that defines the key for each {@code FrequencyType} option.
             *
             * @param key the integer value representing each {@code FrequencyType} option.
             */
            FrequencyType(String key) {
                this.key = key;
            }

            /**
             * Get the integer representation for the {@code FrequencyType} option.
             *
             * @return integer representation of the {@code FrequencyType} option.
             */
            public String getKey() {
                return this.key;
            }

            /**
             * Get the {@code FrequencyType} enum value corresponding to an integer representation.
             *
             * @param key the integer representation of the {@code FrequencyType} enum.
             *
             * @return The {@code FrequencyType} enum for the integer representation.
             */
            public static FrequencyType fromKey(String key) {
                for(FrequencyType type : FrequencyType.values()) {
                    if(type.getKey().equals(key)) {
                        return type;
                    }
                }
                return ONCE;
            }

        }
    }
}
