package com.flybits.context.utils;


import androidx.annotation.NonNull;
import com.flybits.context.models.Rule;
import com.flybits.context.models.RulePredicate;
import com.flybits.context.models.RulePredicateQuery;
import org.json.JSONException;
import org.json.JSONObject;

import java.security.InvalidParameterException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * The {@code RuleBuilder} class is responsible for creating the appropriate JSON request body based
 * on the attributes set by the application. Using the JSON request body the creation of a
 * {@link Rule} is made possible.
 */
public class RuleBuilder {

    private String name;
    private String description;
    private boolean defaultEvaluated;
    private String color;
    private String scope;
    private String stringRepresentation;
    private static final String HEX_PATTERN_NAME = "[^A-Za-z0-9]";
    private HashMap<String, String> mapOfPredicateData;
    private RuleScheduler scheduler;

    private RuleBuilder(Builder builder) {
        this.name                   = builder.name;
        this.defaultEvaluated       = builder.defaultEvaluated;
        this.color                  = builder.color;
        this.scope                  = builder.scope;
        this.stringRepresentation   = builder.stringRepresentation;
        this.mapOfPredicateData     = builder.mapOfPredicateData;
        if (builder.description != null){
            this.description    = builder.description;
        }
        if (builder.color != null){
            this.color    = builder.color;
        }
        if (builder.scheduler != null){
            this.scheduler  = builder.scheduler;
        }
    }

    /**
     * The network request body that should be used when creating a {@link Rule} through a network
     * request.
     *
     * @return The JSON String that should be used as the request body for the network request when
     * creating a {@link Rule}.
     */
    public String getRequestBody() {

        JSONObject object = new JSONObject();
        try {
            object.put("name", name);
        } catch (JSONException e) {}
        try {
            object.put("defaultEvaluated", defaultEvaluated);
        } catch (JSONException e) {}
        try {
            object.put("scope", scope);
        } catch (JSONException e) {}

        try{
            String ruleRepresentation = convertNameToProperStructure(name) + "() :- " + stringRepresentation;
            object.put("stringRepresentation", ruleRepresentation);
        } catch (JSONException e) {}

        try{
            if (description != null) {
                object.put("description", description);
            }
        } catch (JSONException e) {}

        try{
            if (color != null){
                object.put("color", color);
            }
        } catch (JSONException e) {}

        try{
            if (scheduler != null){
                object.put("schedule", scheduler.toJSONObject());
            }
        } catch (JSONException e) {}

        try {
            if (mapOfPredicateData != null && mapOfPredicateData.size() > 0) {

                JSONObject mapPredicateData = new JSONObject();
                Iterator it = mapOfPredicateData.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<String, String> pair = (Map.Entry<String, String>) it.next();
                    mapPredicateData.put(pair.getKey(), pair.getValue());
                    it.remove();
                }

                object.put("pluginData", mapPredicateData);
            }
        }catch (JSONException e){}

        return object.toString();
    }

    private String convertNameToProperStructure(String nameAsString){
        return nameAsString.replaceAll(HEX_PATTERN_NAME, "-");
    }

    /**
     * The {@code Builder} class that defines all the possible filter options that can be used when
     * creating a {@link Rule}.
     */
    public static class Builder {

        private String name;
        private String description;
        private boolean defaultEvaluated;
        private String color;
        private String scope;
        private String stringRepresentation;
        private Pattern patternColor;
        private HashMap<String, String> mapOfPredicateData;
        private RuleScheduler scheduler;

        private static final String HEX_PATTERN_COLOR = "^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$";

        private Builder(){
            this.scope              = "user";
            this.defaultEvaluated   = false;

            patternColor = Pattern.compile(HEX_PATTERN_COLOR);
        }

        /**
         * Constructor that defines a {@code Builder} that allows options to be added to it to make
         * the {@code Rule} construction process more simple for the application. This constructor
         * only needs a {@code name} and {@link RulePredicateQuery} for the {@code Rule} to be
         * created.
         *
         * @param name The name of the rule. A valid name consists of a alphanumeric characters
         *             without any spaces.
         * @param predicateQuery The {@link RulePredicateQuery} which is used to construct a rule
         *                  query.
         */
        public Builder(@NonNull String name, @NonNull RulePredicateQuery predicateQuery){
            this();
            this.name                   = name;
            this.stringRepresentation   = predicateQuery.getQuery();
            this.mapOfPredicateData     = predicateQuery.getMapOfRulePredicate();
        }

        /**
         * Constructor that defines a {@code Builder} that allows options to be added to it to make
         * the {@code Rule} construction process more simple for the application. This constructor
         * only needs a {@code name} and {@link RulePredicate} for the {@code Rule} to be created.
         *
         * @param name The name of the rule. A valid name consists of a alphanumeric characters
         *             without any spaces.
         * @param predicate The {@link RulePredicate} which is used to construct a rule query. This
         *                  is a useful shutcut to {@link RulePredicateQuery} as it requires less
         *                  work for a simple 1 predicate query.
         */
        public Builder(@NonNull String name, @NonNull RulePredicate predicate){
            this();
            this.name                   = name;
            this.stringRepresentation   = "("+predicate.getString()+")";
            this.mapOfPredicateData     = predicate.getPluginData();
        }

        /**
         * Build the {@code RuleBuilder} object to be used for constructing a {@link Rule} to be
         * processed for the logged in {@code User}.
         *
         * @return {@code RuleBuilder} object which contains all the information needed to
         * successfully construct a rule.
         */
        public RuleBuilder build(){
            return new RuleBuilder(this);
        }

        /**
         * Set the {@link Rule} color.
         *
         * @param hex The color that is associated to the {@link Rule} as a hex string. Accepted hex
         *            values include #XXX or #XXXXXX when X is a Hexadecimal value.
         *
         * @return A {@code Builder} object where additional options can be set.
         */
        public Builder setColor(String hex) throws InvalidParameterException{

            Matcher matcher = patternColor.matcher(hex);
            if (matcher.matches()){
                this.color  = hex;
            }else{
                throw new InvalidParameterException("The color parameter must be in either of the following formats " +
                        "#XXXXXX or #XXX where X is Hex value");
            }
            return this;
        }

        /**
         * Sets the default evaluation value as true instead of the default false.
         *
         * @return A {@code Builder} object where additional options can be set.
         */
        public Builder setDefaultEvaluationAsTrue(){
            this.defaultEvaluated   = true;
            return this;
        }

        /**
         * Set the {@link Rule} description and what is accomplished through the rule.
         *
         * @param description The description that should be describe what the {@link Rule} is
         *                    trying to accomplish.
         * @return A {@code Builder} object where additional options can be set.
         */
        public Builder setDescription(String description){
            this.description    = description;
            return this;
        }

        /**
         * Set the {@link RuleScheduler} object associated to the {@link Rule}. This
         * {@link RuleScheduler} indicates that {@link Rule}s should only be processed during the
         * times indicated within the {@link RuleScheduler}.
         *
         * @param scheduler The {@link RuleScheduler} object associated to the {@link Rule}.
         * @return A {@code Builder} object where additional options can be set.
         */
        public Builder setScheduler(RuleScheduler scheduler){
            if (scheduler != null) {
                this.scheduler = scheduler;
            }
            return this;
        }
    }

}
