package com.flybits.context.utils;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.os.Build;
import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import com.flybits.context.FlyingContextConstants;
import com.flybits.context.models.BasicData;
import com.flybits.context.models.ContextData;

import java.util.List;

/**
 * The {@code ContextUtilities} class is responsible for performing utility functions that are used
 * throughout the Context SDK and simplify the process of executing specific code segments.
 */
public class ContextUtilities {

    /**
     * Broadcast updated {@link ContextData} that has been changed from a previous value to anyone
     * who is listening.
     *
     * @param mContext The current of the state of the application.
     * @param data The new updated {@link ContextData}.
     * @param pluginID The unique plugin identifier that represents a specific {@code Plug-in}.
     * @param timeOccurred The time (in seconds) that the {@code data} was updated. This is based
     *                     on local device time.
     */
    public static void broadcastContextUpdate(Context mContext, ContextData data, String pluginID, long timeOccurred){

        Intent contextData  = new Intent(FlyingContextConstants.BROADCAST_CONTEXT_UPDATE);
        contextData.putExtra(FlyingContextConstants.BROADCAST_CONTEXT_UPDATE_EXTRA_PLUGIN, pluginID);
        contextData.putExtra(FlyingContextConstants.BROADCAST_CONTEXT_UPDATE_EXTRA_TIME, timeOccurred);
        contextData.putExtra(FlyingContextConstants.BROADCAST_CONTEXT_UPDATE_EXTRA_DATA, data);
        mContext.sendBroadcast(contextData);

        contextData.setAction(pluginID);
        mContext.sendBroadcast(contextData);
    }

    /**
     * Checks to see if the necessary permission has been added to the AndroidManifest.xml file.
     * This could include either;
     *
     * &lt;p&gt;
     *     &lt;ul&gt;
     *         &lt;li&gt;&lt;uses-permission android:name="..." /&gt;&lt;/li&gt;
     *         &lt;li&gt;&lt;permission android:name="..." /&gt;&lt;/li&gt;
     *     &lt;/ul&gt;
     * &lt;/p&gt;
     *
     * @param context The current state of the application.
     * @param permission The name of the permission that should be checked to make sure is declared
     *                   in the AndroidManifest.xml file.
     *
     * @return true if the {@link android.app.Activity} is declared in the AndroidManifest.xml,
     * false otherwise.
     */
    public static boolean isPermissionGranted(@NonNull Context context, @NonNull String permission){

        if (context == null || permission == null){
            return false;
        }

        if (Build.VERSION.SDK_INT > Build.VERSION_CODES.LOLLIPOP_MR1) {
            if (ContextCompat.checkSelfPermission(context, permission) == PackageManager.PERMISSION_GRANTED){
                return true;
            }
        }
        return hasPermission(context, permission);
    }

    /**
     * Checks to see if an {@link android.app.Service} has been defined in the application's
     * AndroidManifest.xml file.
     *
     * @param context The current state of the application.
     * @param intent The intent that holds the {@link android.app.Service} class information that
     *               needs to be check to see if it is declared with the application's
     *               AndroidManifest.xml file.
     *
     * @return true if the {@link android.app.Service} is declared in the AndroidManifest.xml,
     * false otherwise.
     */
    public static boolean isServiceDefined(@NonNull Context context, @NonNull Intent intent) {
        if (context == null || intent == null){
            return false;
        }

        List<ResolveInfo> list = context.getPackageManager().queryIntentServices(intent,
                PackageManager.MATCH_DEFAULT_ONLY);
        return list.size() > 0;
    }


    /**
     * Retrieves the JSON representation of the data stored with in the Context Database.
     *
     * @param dataPlugins The {@link BasicData} that should have a JSON Representation created for.
     *
     * @return The JSON representation of the current context state.
     */
    public static String getContextData(@NonNull List<BasicData> dataPlugins){

        if (dataPlugins == null){
            return "[]";
        }

        String json = "[";

        for (int i = 0; i < dataPlugins.size(); i++) {
            String dataToSendJson = dataPlugins.get(i).toString();
            json += dataToSendJson;
            if (i < dataPlugins.size() - 1){
                json += ",";
            }
        }

        json += "]";
        return json;
    }

    private static boolean hasPermission(Context context, String permission) {
        try {
            PackageManager manager = context.getPackageManager();
            if (manager != null) {
                PackageInfo info =  manager.getPackageInfo(context.getPackageName(), PackageManager.GET_PERMISSIONS);
                if (info.requestedPermissions != null) {
                    for (String p : info.requestedPermissions) {
                        if (p.equals(permission)) {
                            return true;
                        }
                    }
                }
            }
        } catch (Exception e) {}
        return false;
    }
}
