package com.flybits.context.plugins.location;

import android.location.Location;
import android.os.Parcel;
import android.os.Parcelable;
import com.flybits.context.models.ContextData;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * The {@code LocationData} indicates the physical location of the user based on latitude/longitude
 * coordinates. Additionally, information about the user such as their altitude, bearing, and speed
 * is also indicated.
 */
public class LocationData extends ContextData implements Parcelable {

    private double lat;
    private double lng;
    private double altitude;
    private float speed;
    private float bearing;
    private int timeZone;

    /**
     * Default constructor needed for generics instantiation
     */
    public LocationData(){}

    /**
     * Constructor used for un-flattening a {@code LocationData} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code LocationData} parcel.
     */
    protected LocationData(Parcel in){
        super();
        this.lat            = in.readDouble();
        this.lng            = in.readDouble();
        this.altitude       = in.readDouble();
        this.bearing        = in.readFloat();
        this.speed          = in.readFloat();
        this.timeZone       = in.readInt();
        setTime(in.readLong());
    }

    /**
     * Constructor used to define the properties of the {@code LocationData} based on
     * {@link Location} which is obtained from the GPS sensor or other internal mechanisms.
     *
     * @param location The {@link Location} of the user.
     */
    public LocationData(Location location){
        super();
        this.lat            = location.getLatitude();
        this.lng            = location.getLongitude();
        this.altitude       = location.getAltitude();
        this.bearing        = location.getBearing();
        this.speed          = location.getSpeed();
    }

    /**
     * Constructor used to define the {@code LocationData} based on the JSON representation of the
     * location data.
     *
     * @param json The JSON representation of a location.
     */
    public LocationData(String json){
        super();
        fromJson(json);
    }

    /**
     * Get the altitude if available, in meters. If this location does not have an altitude then 0.0
     * will be returned.
     *
     * @return The altitude of the user, if no altitude can be detected then 0.0 is returned.
     */
    public double getAltitude() {
        return altitude;
    }

    /**
     * Get the bearing, in degrees. It is a range between 0.0 - 360.0 and indicates the device
     * orientation. If a location does not have a bearing then 0.0 will be returned.
     *
     * @return The bearing of the user, if no bearing can be detected then 0.0 is returned.
     */
    public float getBearing() {
        return bearing;
    }

    /**
     * Get the latitude coordinates in degrees.
     *
     * @return The latitude of the user.
     */
    public double getLatitude() {
        return lat;
    }

    /**
     * Get the longitude coordinates in degrees.
     *
     * @return The longitude of the user.
     */
    public double getLongitude() {
        return lng;
    }

    /**
     * Get the speed in meters/second. If this location does not have a speed then 0.0 will be
     * returned.
     *
     * @return The speed of the user, if no speed can be detected then 0.0 is returned.
     */
    public float getSpeed(){
        return speed;
    }

    /**
     * Get the timezone based on the user's location.
     *
     * @return The integer representing the timezone of the user's location.
     */
    public int getTimeZone() {
        return timeZone;
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.location";
    }

    @Override
    public String toJson() {

        JSONObject object=new JSONObject();
        try {
            object.put("lat", lat);
            object.put("lng", lng);
            object.put("altitude", altitude);
            object.put("bearing", bearing);
        }catch (JSONException exception){}

        return object.toString();
    }

    @Override
    public void fromJson(String json) {
        try {
            JSONObject jsonObj  = new JSONObject(json);
            lat                 = jsonObj.getDouble("lat");
            lng                 = jsonObj.getDouble("lng");
            altitude            = jsonObj.getDouble("altitude");
            bearing             = (float) jsonObj.getDouble("bearing");
            timeZone            = jsonObj.getInt("timeZone");

        }catch (JSONException exception){}
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        LocationData that = (LocationData) o;

        if (lat != that.lat) return false;
        if (lng != that.lng) return false;
        if (altitude != that.altitude) return false;
        if (bearing != that.bearing) return false;
        return timeZone == that.timeZone;
    }

    @Override
    public String toString(){
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeDouble(lat);
        dest.writeDouble(lng);
        dest.writeDouble(altitude);
        dest.writeFloat(bearing);
        dest.writeFloat(speed);
        dest.writeInt(timeZone);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code LocationData} objects
     */
    public static final Creator<LocationData> CREATOR = new Creator<LocationData>() {
        public LocationData createFromParcel(Parcel in) {
            return new LocationData(in);
        }

        public LocationData[] newArray(int size) {
            return new LocationData[size];
        }
    };
}
