package com.flybits.context.models;

import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcel;
import android.os.Parcelable;
import androidx.annotation.NonNull;
import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.deserializations.DeserializePagedResponse;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.ContextScope;
import com.flybits.context.deserializations.DeserializeRule;
import com.flybits.context.models.results.RulesResult;
import com.flybits.context.utils.RuleBuilder;
import com.flybits.context.utils.RulesQueryParameters;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * This class is used to represent a Context {@code Rule} within Flybits. A rule can be either
 * evaluated as true or false depending on values obtained from the Context Plugins which can be
 * obtained from a local sensor or an external one. A {@code Rule} needs to contain at least one
 * condition which needs to be satisfied to be true.
 *
 * <p>For example, an application developer may create a Context {@link Rule}
 * "Tired" that is comprised of 2 context plugins; Step Counter and Sleep Quality.</p>
 */
public class Rule implements Parcelable {

    public static final String API = ContextScope.ROOT+"/rules";

    private String id;
    private String name;
    private String description;
    private String color;
    private String stringRepresentation;
    private RuleScope scope;
    private String creatorID;
    private boolean defaultEvaluation;

    /**
     * Constructor used for un-flattening a {@code RuleEvalutated} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code RuleEvalutated} parcel.
     */
    public Rule(Parcel in){
        id                      = in.readString();
        name                    = in.readString();
        description             = in.readString();
        color                   = in.readString();
        stringRepresentation    = in.readString();
        scope                   = RuleScope.fromKey(in.readString());
        creatorID               = in.readString();
        defaultEvaluation       = in.readInt() == 1;
    }

    /**
     * Constructor used for creating a {@code Rule}.
     *
     * @param id The unique identifier for {@code Rule}.
     * @param name The name of the {@code Rule}.
     * @param description The description of the {@code Rule}.
     * @param color The color representation of the {@code Rule}.
     * @param stringRepresentation The String query that represents which Context Plugins and under
     *                             what conditions the {@code Rule} should be evaluated as true.
     * @param scope The scope indicates whether or not this {@code Rule} should only be evaluated
     *              for this user or for all user\'s within the Project.
     * @param ownerID The unique identifier of the user who created this {@code Rule}.
     * @param defaultEvaluation Indicates whether or not the default value of the {@code Rule} is
     *                          true or false.
     */
    public Rule(String id, String name, String description, String color, String stringRepresentation,
                String scope, String ownerID, boolean defaultEvaluation) {
        this.id                     = id;
        this.name                   = name;
        this.description            = description;
        this.color                  = color;
        this.stringRepresentation   = stringRepresentation;
        this.scope                  = RuleScope.fromKey(scope);
        this.creatorID              = ownerID;
        this.defaultEvaluation      = defaultEvaluation;
    }

    /**
     * Get the color representation of the {@code Rule}. This is an optional attribute and therefore
     * it can be null.
     *
     * @return The string representation of the color. This value should be either #XXX or #XXXXXX
     * where X is a hexadecimal value.
     */
    public String getColor() {
        return color;
    }

    /**
     * Get the unique identifier of the user who created the {@code Rule}.
     *
     * @return The unqiue identifier of the user who created the {@code Rule}.
     */
    public String getCreatorID() {
        return creatorID;
    }

    /**
     * Get the description of the {@code Rule}. This is an optional attribute and therefore it
     * can be null.
     *
     * @return The description of the {@code Rule} or null if the description is not set.
     */
    public String getDescription() {
        return description;
    }

    /**
     * Get the unique identifier of the {@code Rule}.
     *
     * @return The unique identifier of the {@code Rule}.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the name of the {@code Rule}.
     *
     * @return The name of the {@code Rule}.
     */
    public String getName() {
        return name;
    }

    /**
     * Get the scope of the {@code Rule}. The scope can either be "user" which means that this
     * {@code Rule} is only evaluated for the user that created the {@code Rule}, or "tenant" which
     * means that this {@code Rule} is evaluated for everyone in the Project.
     *
     * @return The {@link RuleScope} of the {@code Rule}.
     */
    public RuleScope getScope() {
        return scope;
    }

    /**
     * Get the String representation of the {@code Rule} query that is used to define when a
     * {@code Rule} is evaluated.
     *
     * @return The String representation of the {@code Rule}.
     */
    public String getStringRepresentation() {
        return stringRepresentation;
    }

    /**
     * Get the default evaluation of the {@code Rule}.
     *
     * @return true if the {@code Rule}'s default evaluation is set to true, false otherwise.
     */
    public boolean isDefaultEvaluation() {
        return defaultEvaluation;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Rule rule = (Rule) o;

        return !(id != null ? !id.equals(rule.id) : rule.id != null);
    }

    /**
     * Describe the kinds of special objects contained in this Parcelable's marshalled
     * representation.
     *
     * @return a bitmask indicating the set of special object types marshalled by the Parcelable.
     */
    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public String toString(){
        return "[ Id: " + id + ", name: " + name + " ]";
    }

    /**
     * Flatten this {@code GeoPoint} into a Parcel.
     *
     * @param out The Parcel in which the {@code GeoPoint} object should be written.
     * @param flags Additional flags about how the DateOfBirth object should be written.
     * May be 0 or {@link #PARCELABLE_WRITE_RETURN_VALUE}.
     */
    @Override
    public void writeToParcel(Parcel out, int flags) {
        out.writeString(id);
        out.writeString(name);
        out.writeString(description);
        out.writeString(color);
        out.writeString(stringRepresentation);
        out.writeString(scope.getKey());
        out.writeString(creatorID);
        out.writeInt(defaultEvaluation ? 1 : 0);
    }

    /**
     * Parcelable.Creator that instantiates {@code Rule} objects
     */
    public static final Creator<Rule> CREATOR = new Creator<Rule>(){
        public Rule createFromParcel(Parcel in){
            return new Rule(in);
        }

        public Rule[] newArray(int size){
            return new Rule[size];
        }
    };

    /**
     * Create this {@code Rule} based on the attributes defined within the {@link RuleBuilder}.
     *
     * @param mContext The context of the application.
     * @param newRule The {@link RuleBuilder} used to define the attributes of the {@code Rule}.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code ObjectResult} that contains the network request for creating {@code Rule}.
     */
    public static ObjectResult<Rule> create(final @NonNull Context mContext, final @NonNull RuleBuilder newRule,
                                            @NonNull ObjectResultCallback<Rule> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<Rule> request = new ObjectResult<Rule>(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    final Result<Rule> created = FlyAway.post(mContext, API, newRule.getRequestBody(),
                            new DeserializeRule(), "Rule.create", Rule.class);
                    request.setResult(created);
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * Delete this {@code Rule} based on the identifier set through the
     * {@link #Rule(String, String, String, String, String, String, String, boolean)} constructor.
     *
     * @param mContext The context of the application.
     * @param callback The callback that is used to indicate whether or the network request was
     *                 successful or not.
     *
     * @return The {@code BasicResult} that contains the network request for deleting a specific
     * {@code Rule}.
     */
    public BasicResult delete(@NonNull final Context mContext, @NonNull BasicResultCallback callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult request = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {
                try{
                    final Result deleted = FlyAway.delete(mContext, API, "Rule.delete", id);
                    request.setResult(deleted);
                }catch (final FlybitsException e){
                    request.setFailed(e);
                }
            }
        });
        return request;
    }

    /**
     * A request to get a list of {@code Rule}s based on the {@code QueryParameters} parameter which
     * defines which options should be used to filter the list of returned parameters.
     *
     * @param mContext The context of the application.
     * @param params The {@code QueryParameters} that indicate filters that should be placed on
     *               the request.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     *
     * @return The {@code RulesResult} that contains the network request for getting a list of
     * {@code Rule}s.
     */
    public static RulesResult get(@NonNull final Context mContext, @NonNull final RulesQueryParameters params,
                                  @NonNull PagedResultCallback<Rule> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final RulesResult result = new RulesResult(mContext, params, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeRule singleDeserializaer             = new DeserializeRule();
                    DeserializePagedResponse<Rule> deserializer     = new DeserializePagedResponse<Rule>(singleDeserializaer);
                    final Result<PagedResponse<Rule>> getRules = FlyAway.get(mContext, API, params, deserializer, "Rule.get");
                    result.setResult(getRules);
                }catch (final FlybitsException e){
                    result.setResult(new Result<PagedResponse<Rule>>(e,"Rule.get() failed"));
                }
            }
        });
        return result;
    }
}