package com.flybits.context.models;

import androidx.room.Ignore;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.os.Parcelable;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.ContextManager;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.utils.ContextUtilities;
import com.flybits.internal.DeboucerContextReporting;
import com.flybits.internal.Debouncer;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code ContextData} abstract class is used to define an object that is considered
 * "context-ready". This means that the Context Server will be able to process the object that is
 * sent to it as it will have implemented the necessary methods.
 */
public abstract class ContextData implements Parcelable {

    @Ignore
    private long    timeInSeconds;

    public ContextData(){
        setTime((System.currentTimeMillis() / 1000));
    }

    /**
     * Method that indicates whether two pieces of the same {@code ContextData} are identical. This
     * is important because if two objects are the same then the SDK only wants to update the last
     * modified field rather than simply add it to the Context information database.
     *
     * @param o The object used to compare with the current {@code ContextData}.
     * @return true if the current object and the {@code o} object are the same, false otherwise.
     */
    public abstract boolean equals(Object o);

    /**
     * Method to transform a JSON representation of a {@code ContextData} into the variables of
     * the {@link com.flybits.context.plugins.FlybitsContextPlugin}.
     *
     * @param json JSON String representation of a {@code ContextData}.
     */
    public abstract void fromJson(String json);

    /**
     * Return the time, in seconds, that indicate when the {@code ContextData} was obtained.
     *
     * @return The time in seconds.
     */
    public long getTime(){
        return timeInSeconds;
    }

    /**
     * Returns the context plugin id that this data is for.
     *
     * @return The plugin id.
     */
    public abstract String getPluginID();

    /**
     * Sets the device time that indicates when specific {@code ContextData} was retrieved.
     *
     * @param time The time in seconds that represents when time was obtained.
     */
    public void setTime(long time){
        timeInSeconds = time;
    }

    /**
     * Method to transform the {@link com.flybits.context.plugins.FlybitsContextPlugin}'s variables
     * to a JSON format so that it can be re-initialized when the device re-starts.
     *
     * @return The JSON representation of the {@link com.flybits.context.plugins.FlybitsContextPlugin}.
     */
    public abstract String toJson();

    /**
     * Updates the context value for a specific {@code Context Plug-in} based on its
     * {@link ContextData} value. This method will be executed at most only once every 30 seconds in
     * order to reduce the amount of network requests that happen.
     *
     * @param context The context of the application.
     * @param timeInSec The time in epoch seconds of when the data retrieval took place.
     */
    public void update(final Context context, final long timeInSec) {
        final ExecutorService executorService   = Executors.newSingleThreadExecutor();
        executorService.execute(new Runnable() {
            public void run() {
                if (updateDB(context, timeInSec)){
                    ContextUtilities.broadcastContextUpdate(context, ContextData.this, getPluginID(), timeInSec);
                    DeboucerContextReporting task = new DeboucerContextReporting(context);
                    Debouncer.getInstance().call(task);
                }
            }
        });
    }

    /**
     * Updates the context value for a specific {@code Context Plug-in} based on its
     * {@link ContextData} value. This method will be executed at most only once every 30 seconds in
     * order to reduce the amount of network requests that happen.
     *
     * @param context The context of the application.
     * @param timeInSec The time in epoch seconds of when the data retrieval took place.
     * @param callback The {@code BasicResultCallback} that indicates whether or not the request was
     *                 successful or not. If not, a reason will be provided.
     * @return The {@code BasicResult} which contains information about the request being made.
     * @deprecated Starting in version 1.9.0 please use {@link #update(Context, long)}.
     */
    @Deprecated
    public BasicResult update(final Context context, final long timeInSec, final BasicResultCallback callback) {

        update(context, timeInSec);

        final Handler handler                   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService   = Executors.newSingleThreadExecutor();
        final BasicResult basicResult = new BasicResult(callback, handler, executorService);
        Result result = new Result(200, "");
        basicResult.setResult(result);
        return basicResult;
    }

    /**
     * Updates the context value for a specific {@code Context Plugin} based on its
     * {@link ContextData} value. This method does not have a 30 second restriction and should only
     * be used for Development Builds for testing otherwise your Account will be blocked.
     *
     * @param context The context of the application.
     * @param callback The {@code BasicResultCallback} that indicates whether or not the request was
     *                 successful or not. If not, a reason will be provided.
     * @return The {@code BasicResult} which contains information about the request being made.
     */
    public BasicResult updateNow(final Context context, final BasicResultCallback callback) {

        final Handler handler                   = new Handler(Looper.getMainLooper());
        final ExecutorService executorService   = Executors.newSingleThreadExecutor();
        final BasicResult basicResult           = new BasicResult(callback, handler, executorService);
        executorService.execute(new Runnable() {
            public void run() {

                long currentTime = System.currentTimeMillis()/1000;
                if (updateDB(context, currentTime)){
                    ContextUtilities.broadcastContextUpdate(context, ContextData.this, getPluginID(), currentTime);
                    ContextManager.flushContextData(context, new BasicResultCallback() {
                        @Override
                        public void onSuccess() {
                            Result result = new Result(200, "");
                            basicResult.setResult(result);
                        }

                        @Override
                        public void onException(FlybitsException exception) {
                            Result result = new Result(exception, "");
                            basicResult.setResult(result);
                        }
                    });
                }else {

                    Result result = new Result(200, "");
                    basicResult.setResult(result);
                }
            }
        });
        return basicResult;
    }

    protected boolean updateDB(Context context, long timeInSeconds){

        String pluginID = getPluginID();
        BasicData dataFromCursor   = ContextDatabase.getDatabase(context).basicDataDao().get(pluginID);
        if (dataFromCursor != null) {

            if (dataFromCursor.getValueAsString() != null
                    && (!dataFromCursor.getValueAsString().equals(toJson())
                    || !dataFromCursor.isSent())) {

                BasicData tempData  = ContextDatabase.getDatabase(context).basicDataDao().get(pluginID);
                tempData.setTimestamp(timeInSeconds);
                tempData.setValueAsString(toJson());
                tempData.setSent(false);

                Logger.appendTag("ContextData").i("Update: " + toJson());
                ContextDatabase.getDatabase(context).basicDataDao().update(tempData);
                return true;
            }
        }else{
            BasicData dataBasic                 = new BasicData(pluginID, timeInSeconds, false, toJson());
            Logger.appendTag("ContextData").i("Insert: " + toJson());
            ContextDatabase.getDatabase(context).basicDataDao().insert(dataBasic);
            return true;
        }
        return false;
    }

    @Override
    public String toString(){
        return toJson();
    }
}
