package com.flybits.context.db

import androidx.sqlite.db.SupportSQLiteDatabase
import androidx.room.Database
import androidx.room.Room
import androidx.room.RoomDatabase
import androidx.room.TypeConverters
import androidx.room.migration.Migration
import android.content.Context
import com.flybits.context.db.converters.BeaconTypeConverter
import com.flybits.context.db.converters.ServiceTypeConverter
import com.flybits.context.db.dao.*
import com.flybits.context.models.BasicData
import com.flybits.context.models.internal.Plugin
import com.flybits.context.plugins.beacon.BeaconActive
import com.flybits.context.plugins.beacon.BeaconMonitored
import com.flybits.context.models.RuleEvalutated
import com.flybits.context.plugins.FlybitsContextPlugin

@Database(entities = [Plugin::class, RuleEvalutated::class, BasicData::class, BeaconActive::class, BeaconMonitored::class, FlybitsContextPlugin::class], version = 3)
@TypeConverters(BeaconTypeConverter::class, ServiceTypeConverter::class)
abstract class ContextDatabase : RoomDatabase() {

    abstract fun pluginDao(): DBPluginDAO
    abstract fun ruleDao(): RuleEvaluationDAO
    abstract fun basicDataDao(): BasicDataDAO
    abstract fun beaconActiveDao(): BeaconActiveDAO
    abstract fun beaconMonitoredDao(): BeaconMonitoredDAO
    abstract fun flybitsContextPluginDAO(): FlybitsContextPluginDAO

    companion object {

        @Volatile
        private var INSTANCE: ContextDatabase? = null

        @JvmStatic
        fun getDatabase(context: Context): ContextDatabase {
            return INSTANCE ?: synchronized(this) {
                INSTANCE ?: Room.databaseBuilder(context, ContextDatabase::class.java, "flybits-context-db")
                        .addMigrations(MIGRATION_2_3)
                        .fallbackToDestructiveMigration()
                        .build().also { INSTANCE = it }
            }
        }

        fun destroyInstance() {
            INSTANCE = null
        }

        internal val MIGRATION_2_3 = object: Migration(2,3) {
            override fun migrate(database: SupportSQLiteDatabase) {
                database.execSQL("CREATE TABLE ${FlybitsContextPlugin.TABLE} " +
                        "(${FlybitsContextPlugin.COLUMN_SERVICE} TEXT PRIMARY KEY NOT NULL" +
                        ", ${FlybitsContextPlugin.COLUMN_IS_RUNNING} INTEGER NOT NULL" +
                        ", ${FlybitsContextPlugin.COLUMN_REFRESH_TIME} INTEGER NOT NULL" +
                        ", ${FlybitsContextPlugin.COLUMN_REFRESH_TIME_FLEX} INTEGER NOT NULL)" )
            }
        }
    }

}