package com.flybits.context;

import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.BasicResult;
import com.flybits.commons.library.api.results.ObjectResult;
import com.flybits.commons.library.api.results.callbacks.BasicResultCallback;
import com.flybits.commons.library.api.results.callbacks.ObjectResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.http.RequestStatus;
import com.flybits.commons.library.logging.Logger;
import com.flybits.commons.library.models.User;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.db.ContextDatabase;
import com.flybits.context.models.BasicData;
import com.flybits.context.models.ContextData;
import com.flybits.context.models.ContextPriority;
import com.flybits.context.plugins.ContextPlugin;
import com.flybits.context.plugins.FlybitsContextPlugin;
import com.flybits.context.services.FlybitsContextPluginService;
import com.flybits.context.services.FlybitsContextPluginsWorker;
import com.flybits.context.utils.ContextUtilities;
import com.flybits.internal.db.CommonsDatabase;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.OneoffTask;
import com.google.android.gms.gcm.PeriodicTask;
import com.google.android.gms.gcm.Task;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

public class ContextManager {

    private static final String API_CONTEXT_DATA = ContextScope.ROOT + "/ctxdata";
    private static final String TAG_CM = "ContextManager";

    /**
     * Get Context Plug-in data that is stored within the Flybits Context DB. It's important to note
     * that {@code pluginID} and {@code subclass} must match, otherwise an empty {@link ContextData}
     * object will be returned or null if there was an error.
     *
     * @param context  The context of the application.
     * @param pluginID The unique identifier that represents a specific {@link ContextPlugin}.
     * @param subclass The class that should be instantiated and context data should be parsed into.
     * @param callback The {@code ObjectResultCallback} that indicates whether or not the data for
     *                 {@code pluginID}, which could null.
     * @param <T>      The generic that represents the a {@link ContextData} class which can parse
     *                 the Context values.
     * @return The {@link ContextData} that the {@link ContextPlugin} data is parsed into.
     */
    public static <T extends ContextData> ObjectResult<T> getData(@NonNull Context context,
                                                                  @NonNull String pluginID,
                                                                  @NonNull Class<T> subclass,
                                                                  ObjectResultCallback<T> callback)
            throws java.security.InvalidParameterException {

        if (context == null) {
            throw new java.security.InvalidParameterException("The Context parameter must not be null");
        }
        if (pluginID == null) {
            throw new java.security.InvalidParameterException("The pluginId parameter must not be null");
        }
        if (subclass == null) {
            throw new java.security.InvalidParameterException("The Class parameter must not be null");
        }

        try {
            T dataTest = (T) subclass.newInstance();
        } catch (Exception e) {
            if (e instanceof InstantiationException) {
                throw new java.security.InvalidParameterException("The .Class file that will be instantiated " +
                        "must contain a constructor with no parameter for this method to be successfully processed");
            }
        }
        return getDataPrivate(context, pluginID, subclass, callback);
    }

    /**
     * Retrieves all the Context information from the local DB and sends it to the server if it is
     * determined that new Context data has been received.
     *
     * @param context  The state of the application.
     * @param callback The {@code BasicResultCallback} that indicates whether or not the request was
     *                 successful or not. If not, a reason will be provided.
     * @return The {@code BasicResult} which contains information about the request being made.
     */
    public static BasicResult flushContextData(@NonNull final Context context, final BasicResultCallback callback) {
        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult basicResult = new BasicResult(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {

                boolean isSuccessful = flushContextData(context);
                Result tempStatus = null;
                if (isSuccessful) {
                    tempStatus = new Result(200, "");
                } else {
                    tempStatus = new Result(new FlybitsException("Error Updating Context"), "");
                }
                final Result resultStatus = tempStatus;
                basicResult.setResult(resultStatus);
            }
        });
        return basicResult;
    }

    /**
     * TODO: Comments
     *
     * @param context
     * @return
     */
    public static boolean flushContextData(@NonNull Context context) {

        if (context == null) {
            return false;
        }

        boolean isSuccessful = false;

        List<BasicData> dataPlugins = ContextDatabase.getDatabase(context).basicDataDao().getAllNotSent();
        String jsonToSend = ContextUtilities.getContextData(dataPlugins);

        if (jsonToSend != null && jsonToSend.length() > 2) {
            try {
                final Result result = FlyAway.post(context, API_CONTEXT_DATA, jsonToSend, null, "FlushUtilities.flushContext", null);
                if (result.getStatus() == RequestStatus.COMPLETED) {

                    for (BasicData data : dataPlugins) {
                        data.setSent(true);
                        ContextDatabase.getDatabase(context).basicDataDao().update(data);
                    }

                    isSuccessful = true;
                } else {
                    Logger.appendTag(TAG_CM).e("Updated FlybitsContext in Server: Failed");
                }
            } catch (FlybitsException e) {
                Logger.exception("FlushUtilities.flushContext", e);
            }
        }

        return isSuccessful;
    }

    /**
     * Refresh the data that is associated to the {@link ContextPlugin}, this may make a network
     * request, open a new activity/dialog, collect information from a sensor and/or anything else
     * that the {@link ContextPlugin} has be designed to do.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be refreshed.
     *                 <p>
     *                 Although this method seems a bit redundant, as it simply call the onRefresh of the Context
     *                 Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void refresh(Context mContext, FlybitsContextPlugin plugin) {
        Executors.newSingleThreadExecutor().execute(() -> refreshSync(mContext, plugin));
    }

    /**
     * Begin the {@link FlybitsContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device only once. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on caller thread.
     * Please Note: This will only create worker to run it once.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be started and to run once.
     */
    public static void refreshSync(Context mContext, FlybitsContextPlugin plugin) {

        User user = CommonsDatabase.getDatabase(mContext).userDao().getActiveUser();
        if (user != null && user.isOptedIn()) {
            plugin.onRefresh(mContext); //state will change here, insert into DAO after
        }
        ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().insert(plugin);
    }

    /**
     * Refresh all the existing {@link FlybitsContextPlugin} to make them run once.
     *
     * @param mContext The Context of the Application.
     * @param callback The callback that indicates all plugins are refreshed.
     */
    public static void refreshAll(Context mContext, final BasicResultCallback callback) {
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final BasicResult result = new BasicResult(callback, new Handler(Looper.getMainLooper()), executorService);
        Executors.newSingleThreadExecutor().execute(() -> refreshAllSync(mContext, result));
    }

    static void refreshAllSync(Context mContext, BasicResult result) {
        User user = CommonsDatabase.getDatabase(mContext).userDao().getActiveUser();
        if (user != null && user.isOptedIn()) {
            List<FlybitsContextPlugin> flybitsContextPluginList
                    = ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().getAll();
            for (FlybitsContextPlugin plugin : flybitsContextPluginList) {
                plugin.onRefresh(mContext); //Run the onetime workers or services for plugins
            }
            result.setSuccess();
        } else {
            FlybitsException e = new FlybitsException("No User has Logged In");
            result.setFailed(e);
        }
    }

    /**
     * Begin the {@link FlybitsContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be started.
     */
    public static void start(Context mContext, FlybitsContextPlugin plugin) {
        Executors.newSingleThreadExecutor().execute(() -> startSync(mContext, plugin));
    }

    /**
     * Begin the {@link FlybitsContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device. If the user is opted out the plugin will be saved and started if the user
     * opts back in. Executes on caller thread.
     * <p>
     * <p>
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be started.
     */
    public static void startSync(Context mContext, FlybitsContextPlugin plugin) {
        User user = CommonsDatabase.getDatabase(mContext).userDao().getActiveUser();
        if (plugin.getContextPluginRetriever().getGenericSuperclass() == FlybitsContextPluginsWorker.class) {
            if (user != null && user.isOptedIn()) {
                plugin.onStart(mContext); //state will change here, insert into DAO after
            }
            ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().insert(plugin);
        } else if (plugin.getContextPluginRetriever().getGenericSuperclass() == FlybitsContextPluginService.class) {
            //Check if the service is added in the manifest.
            Intent intent = new Intent(mContext, plugin.getService());
            if (ContextUtilities.isServiceDefined(mContext, intent)) {
                if (user != null && user.isOptedIn()) {
                    plugin.onStart(mContext); //state will change here, insert into DAO after
                }
                ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().insert(plugin);
            } else {
                Logger.appendTag(TAG_CM).e("Cannot start Context Plugin: In order to start the plugin you need to add the corresponding " + plugin.getService().getSimpleName() + " service in your manifest.");
            }
        }
    }


    /**
     * Resume a {@link FlybitsContextPlugin} that has previously been started.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be resumed.
     */
    static void resume(Context mContext, FlybitsContextPlugin plugin) {
        Executors.newSingleThreadExecutor().execute(() -> resumeSync(mContext, plugin));
    }

    /**
     * Resume a {@link FlybitsContextPlugin} that has previously been started. Executes on
     * the caller thread
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be resumed.
     */
    static void resumeSync(Context mContext, FlybitsContextPlugin plugin) {
        User user = CommonsDatabase.getDatabase(mContext).userDao().getActiveUser();
        resumeSync(mContext, plugin, user);
    }

    /**
     * Resume a {@link FlybitsContextPlugin} that has previously been started. Executes on
     * the caller thread
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be resumed.
     * @param user     The currently authenticated user.
     */
    static void resumeSync(Context mContext, FlybitsContextPlugin plugin, User user) {
        if (plugin.getContextPluginRetriever().getGenericSuperclass() == FlybitsContextPluginsWorker.class) {
            if (user != null && user.isOptedIn()) {
                plugin.onStart(mContext); //state will change here, insert into DAO after
                ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().update(plugin);
            }
        } else if (plugin.getContextPluginRetriever().getGenericSuperclass() == FlybitsContextPluginService.class) {
            Intent intent = new Intent(mContext, plugin.getService());
            //Check if the service is added in the manifest
            if (ContextUtilities.isServiceDefined(mContext, intent)) {
                if (user != null && user.isOptedIn()) {
                    plugin.onStart(mContext); //state will change here, insert into DAO after
                    ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().update(plugin);
                }
            } else {
                ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().delete(plugin);
                Logger.appendTag(TAG_CM).e("Cannot start Context Plugin: In order to start the plugin you need to add the corresponding " + plugin.getService().getSimpleName() + " service in your manifest.");
            }
        }
    }

    /**
     * Stop the {@link FlybitsContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be stopped.
     *                 <p>
     *                 Although this method seems a bit redundant, as it simply call the onStop of the Context
     *                 Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void stop(Context mContext, FlybitsContextPlugin plugin) {
        Executors.newSingleThreadExecutor().execute(() -> stopSync(mContext, plugin));

    }

    /**
     * Stop the {@link FlybitsContextPlugin} that is responsible for collecting Contextual Data about the
     * user or device. Executes on caller thread.
     *
     * @param mContext The Context of the Application.
     * @param plugin   The {@link FlybitsContextPlugin} that should be stopped.
     *                 <p>
     *                 Although this method seems a bit redundant, as it simply call the onStop of the Context
     *                 Plugin it has been written this way to be consistent with the other Flybits platforms
     */
    public static void stopSync(Context mContext, FlybitsContextPlugin plugin) {
        plugin.onStop(mContext); //state will change here, insert into DAO after
        ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().delete(plugin);
    }

    /**
     * Pause a running {@link FlybitsContextPlugin} so that it can be resumed at a later point.
     * Executes on worker thread.
     *
     * @param mContext             The application Context.
     * @param flybitsContextPlugin {@link FlybitsContextPlugin} to be paused.
     */
    public static void pause(Context mContext, FlybitsContextPlugin flybitsContextPlugin) {
        Executors.newSingleThreadExecutor().execute(() -> pauseSync(mContext, flybitsContextPlugin));
    }

    /**
     * Pause a running {@link FlybitsContextPlugin} so that it can be resumed at a later point.
     * Executes on caller thread.
     *
     * @param mContext             The application Context.
     * @param flybitsContextPlugin {@link FlybitsContextPlugin} to be paused.
     */
    public static void pauseSync(Context mContext, FlybitsContextPlugin flybitsContextPlugin) {
        //Don't need to do anything here because it's already in the DAO and can be used during the resume
        flybitsContextPlugin.onStop(mContext); //state will change here, insert into DAO after
        ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().update(flybitsContextPlugin);
    }

    /**
     * Pause all running {@link FlybitsContextPlugin}s.
     *
     * @param mContext The Context of the Application.
     */
    public static void pauseAll(Context mContext) {
        Executors.newSingleThreadExecutor().execute(() -> pauseAllSync(mContext));
    }

    /**
     * Pause all running {@link FlybitsContextPlugin}s.
     *
     * @param mContext The Context of the Application.
     */
    public static void pauseAllSync(Context mContext) {
        List<FlybitsContextPlugin> flybitsContextPluginList
                = ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().getAll();
        for (FlybitsContextPlugin plugin : flybitsContextPluginList) {
            pauseSync(mContext, plugin);
        }
    }

    /**
     * Resume all paused {@link FlybitsContextPlugin}s. Executes on worker thread.
     *
     * @param mContext The Context of the Application.
     */
    public static void resumeAll(Context mContext) {
        Executors.newSingleThreadExecutor().execute(() -> {
            resumeAllSync(mContext);
        });
    }

    /**
     * Resume all paused {@link FlybitsContextPlugin}s. Executes on caller thread.
     *
     * @param mContext The Context of the Application.
     */
    public static void resumeAllSync(Context mContext) {

        User user = CommonsDatabase.getDatabase(mContext).userDao().getActiveUser();
        List<FlybitsContextPlugin> flybitsContextPluginList
                = ContextDatabase.getDatabase(mContext).flybitsContextPluginDAO().getAll();
        for (FlybitsContextPlugin plugin : flybitsContextPluginList) {
            resumeSync(mContext, plugin, user);
        }
    }

    public static void registerForRules(final Context mContext, long timeInSeconds, long timeInSecondsFlex, TimeUnit unit) {

        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setTag("Context_Rules")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setPeriod(unit.toSeconds(timeInSeconds))
                .setService(ContextRulesService.class);
        mGcmNetworkManager.schedule(task.build());
        Logger.appendTag(TAG_CM).d("Activated: Context_Rules, Time to Refresh: " + timeInSeconds);
    }

    static void registerForPluginUpdates(final Context mContext, HashMap<String, String> listOfClass, long timeToRefreshInSec) {

        int listOfClassSize = listOfClass.size();
        SharedPreferences.Editor preferences = ContextScope.getContextPreferences(mContext).edit();
        preferences.putInt(ContextPluginsService.PREF_KEY_NUM_CPS, listOfClassSize);
        preferences.putLong(ContextPluginsService.PREF_KEY_REFRESH_CPS, timeToRefreshInSec);

        Bundle bundle = new Bundle();
        bundle.putInt("listOfItems", listOfClassSize);
        int i = 0;

        Iterator it = listOfClass.entrySet().iterator();
        while (it.hasNext()) {
            Map.Entry<String, String> pair = (Map.Entry<String, String>) it.next();

            String key = pair.getKey();
            bundle.putString("item" + i, key);
            preferences.putString("item" + i, key);

            String value = pair.getValue();
            bundle.putString("class" + i, value);
            bundle.putString("value" + i, value);
            it.remove();
            i++;
        }

        preferences.apply();
        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        OneoffTask oneTimeTask = new OneoffTask.Builder()
                .setService(ContextPluginsService.class)
                .setTag("Context_Plugin_Once")
                .setExecutionWindow(0L, 30L)
                .setRequiredNetwork(Task.NETWORK_STATE_ANY)
                .build();

        mGcmNetworkManager.schedule(oneTimeTask);

        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setExtras(bundle)
                .setTag("Context_Plugin")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setPeriod(timeToRefreshInSec)
                .setService(ContextPluginsService.class);
        mGcmNetworkManager.schedule(task.build());


        Logger.appendTag(TAG_CM).d("Activated: Context_Plugin - Time to Refresh: 90");
    }

    static void registerUploadingContext(final Context mContext, ContextPriority priority, long time, long timeInSecondsFlex, TimeUnit unit) {

        Bundle bundle = new Bundle();
        bundle.putInt("priority", priority.getKey());

        GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
        PeriodicTask.Builder task = new PeriodicTask.Builder()
                .setExtras(bundle)
                .setTag("Context_Uploading")
                .setUpdateCurrent(true)
                .setPersisted(true)
                .setRequiredNetwork(PeriodicTask.NETWORK_STATE_CONNECTED)
                .setPeriod(unit.toSeconds(time))
                .setService(ContextUploadingService.class);
        mGcmNetworkManager.schedule(task.build());
        Logger.appendTag(TAG_CM).d("Activated: Context_Uploading with Priority: " + priority.getKey() + ", Time to Refresh: " + time);
    }

    static void unregisterFromRuleCollection(final Context mContext) {

        Intent intent = new Intent(mContext, ContextRulesService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);

            try {
                mGcmNetworkManager.cancelTask("Context_Rules", ContextRulesService.class);
                Logger.appendTag(TAG_CM).d("UnActivated: Context_Rules");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterFromRuleCollection", ex);
            }
        }
        Logger.appendTag(TAG_CM).d("unregisterFromRuleCollection Completed");
    }

    static void unregisterUploadingContext(final Context mContext) {

        Intent intent = new Intent(mContext, ContextUploadingService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
            try {
                mGcmNetworkManager.cancelTask("Context_Uploading", ContextUploadingService.class);
                Logger.appendTag(TAG_CM).d("UnActivated: Context_Uploading");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterUploadingContext", ex);
            }
        }
        Logger.appendTag(TAG_CM).d("unregisterUploadingContext Completed");
    }

    static void unregisterPluginContext(final Context mContext) {

        Intent intent = new Intent(mContext, ContextPluginsService.class);
        if (ContextUtilities.isServiceDefined(mContext, intent)) {
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
            try {
                mGcmNetworkManager.cancelTask("Context_Plugin", ContextPluginsService.class);
                Logger.appendTag(TAG_CM).d("UnActivated: Context_Plugin");
            } catch (IllegalArgumentException ex) {
                Logger.exception("ContextManager.unregisterPluginContext", ex);
            }
        }
        Logger.appendTag(TAG_CM).d("ContextManager.unregisterPluginContext Completed");
    }

    static @Nullable
    <T extends ContextData> ObjectResult<T> getDataPrivate(@NonNull final Context context,
                                                           @NonNull final String pluginID,
                                                           @NonNull final Class<T> subclass,
                                                           ObjectResultCallback<T> callback) {

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ObjectResult<T> objectResult = new ObjectResult<T>(callback, handler, executorService);

        executorService.execute(new Runnable() {
            public void run() {

                BasicData dataFromCursor = ContextDatabase.getDatabase(context).basicDataDao().get(pluginID);
                if (dataFromCursor != null && dataFromCursor.getValueAsString() != null) {
                    try {
                        T data = (T) subclass.newInstance();
                        data.fromJson(dataFromCursor.getValueAsString());
                        data.setTime(dataFromCursor.getTimestamp());

                        final Result<T> result = new Result<T>(200, "");
                        result.setResponse(data);
                        objectResult.setResult(result);
                    } catch (final Exception e) {
                        objectResult.setResult(new Result(new FlybitsException(e.getMessage()), ""));
                        Logger.exception("ContextManager.getData", e);
                    }
                }
            }
        });
        return objectResult;
    }
}
