package com.flybits.context.plugins.network;

/**
 * <p>The {@code NetworkConnectionType} is used to indicate which type of Internet connection the
 * device currently has (if it can be determined).</p>
 *
 * @author Petar Kramaric
 * @since 2.1.0
 * @version 2.1.0
 */
public enum NetworkConnectionType {

    /**
     * Indicates that the internet connection type could not be determined.
     */
    UNKNOWN(-99),

    /**
     * Indicates that the device is connected to the Internet through 2G, in most cases 2G suggests
     * a slower Internet connection lower than 100 kbps
     */
    _2G(2),

    /**
     * Indicates that the device is connected to the Internet through 3G, in most cases 3G suggests
     * a medium speed Internet connection between 100 kbps - 10 Mbps
     */
    _3G(3),

    /**
     * Indicates that the device is connected to the Internet through 4G, in most cases 4G suggests
     * a fast Internet connection usually above 10 Mbps
     */
    _4G(4),

    /**
     * Indicates that no Internet connection is present therefore no Internet is available on the
     * device.
     */
    NONE(-1),

    /**
     * Indicates that the device is connected to the internet through a WiFi access point. The speed
     * of the Internet varies based on the speed of the WiFi access point.
     */
    WIFI(1);

    private final int key;

    /**
     * Constructor that defines the key for each {@code NetworkConnectionType} option.
     *
     * @param key the integer value representing each {@code NetworkConnectionType} option.
     */
    NetworkConnectionType(int key) {
        this.key = key;
    }

    /**
     * Get the integer representation for the {@code NetworkConnectionType} option.
     *
     * @return integer representation of the {@code NetworkConnectionType} option.
     */
    public int getKey() {
        return this.key;
    }

    /**
     * Get the {@code NetworkConnectionType} enum value corresponding to an integer representation.
     *
     * @param key the integer representation of the {@code NetworkConnectionType} enum.
     *
     * @return The {@code NetworkConnectionType} enum for the integer representation.
     */
    public static NetworkConnectionType fromKey(int key) {
        for(NetworkConnectionType type : NetworkConnectionType.values()) {
            if(type.getKey() == key) {
                return type;
            }
        }
        return UNKNOWN;
    }
}
