package com.flybits.context.plugins.beacon;

import android.os.Parcel;
import android.os.Parcelable;

import com.flybits.context.models.ContextData;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.List;

/**
 * The {@code BeaconDataList} is the object that holds a list of {@link BeaconActive} objects. This
 * is needed because when beacons are sent to the server it needs to send the entire list of beacons
 * on record.
 */
public class BeaconDataList extends ContextData implements Parcelable {

    private List<BeaconActive> data;

    /**
     * Default constructor needed for generics instantiation
     */
    public BeaconDataList(){}

    /**
     * Constructor used for un-flattening a {@code BeaconDataList} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code BeaconDataList} parcel.
     */
    protected BeaconDataList(Parcel in) {
        super();
        data    = new ArrayList<>();
        in.readTypedList(data, BeaconActive.CREATOR);
        setTime(in.readLong());
    }

    /**
     * Constructor used to define the {@code BeaconDataList} based on a preset list of
     * {@link BeaconActive}.
     *
     * @param list The list of {@link BeaconActive}.
     */
    public BeaconDataList(List<BeaconActive> list) {
        super();
        data = list;
    }

    /**
     * Get the list of {@link BeaconActive} that is currently being monitored by the SDK.
     *
     * @return The list of {@link BeaconActive} that are currently being monitored by the SDK.
     */
    public List<BeaconActive> getData() {
        return data;
    }

    @Override
    public boolean equals(Object o) {

        if (o instanceof BeaconDataList) {
            BeaconDataList that = (BeaconDataList) o;
            if (that.data == null && data == null){
                return true;
            }else if ((that.data != null && data == null) || (that.data == null && data != null)){
                return false;
            }
            
            return that.data.equals(data);
        }

        return false;
    }

    @Override
    public void fromJson(String json) {

        data = new ArrayList<>();
        try {
            JSONArray jArr  = new JSONArray(json);
            for(int i = 0; i < jArr.length(); i++)
            {
                JSONObject elementObj = jArr.getJSONObject(i);
                BeaconActive activeBeacon = new BeaconActive(BeaconType.IBeacon, "id1", "id2", "id3", 1);
                activeBeacon.fromJson(elementObj.toString());
                data.add(activeBeacon);
            }
        }catch (JSONException | NullPointerException e){}

    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.beacon";
    }

    @Override
    public String toJson() {

        String jsonToSend = "[";

        for (int i = 0; i < data.size(); i++){

            jsonToSend += data.get(i);
            if (i < data.size() - 1){
                jsonToSend += ",";
            }

        }
        jsonToSend += "]";
        return jsonToSend;
    }

    @Override
    public String toString() {
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeTypedList(data);
        dest.writeLong(getTime());
    }

    /**
     * Parcelable.Creator that instantiates {@code BeaconDataList} objects
     */
    public static final Creator<BeaconDataList> CREATOR = new Creator<BeaconDataList>() {
        @Override
        public BeaconDataList createFromParcel(Parcel in) {
            return new BeaconDataList(in);
        }

        @Override
        public BeaconDataList[] newArray(int size) {
            return new BeaconDataList[size];
        }
    };
}
