package com.flybits.context.plugins;


import android.app.ActivityManager;
import android.app.Notification;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.support.v4.content.ContextCompat;

import com.flybits.commons.library.logging.Logger;
import com.flybits.context.ContextScope;
import com.flybits.context.ReservedContextPlugin;
import com.flybits.context.exceptions.InvalidContextPluginException;
import com.flybits.context.plugins.activity.ActivityContextPluginService;
import com.flybits.context.plugins.battery.BatteryContextPluginService;
import com.flybits.context.plugins.beacon.BeaconScanningService;
import com.flybits.context.plugins.carrier.CarrierContextPluginService;
import com.flybits.context.plugins.fitness.FitnessContextPluginService;
import com.flybits.context.plugins.language.LanguageContextPluginService;
import com.flybits.context.plugins.location.LocationContextPluginService;
import com.flybits.context.plugins.network.NetworkContextPluginService;
import com.flybits.context.plugins.weather.WeatherContextPluginService;
import com.flybits.context.services.FlybitsContextPluginService;
import com.flybits.context.utils.ContextUtilities;
import com.google.android.gms.gcm.GcmNetworkManager;
import com.google.android.gms.gcm.OneoffTask;
import com.google.android.gms.gcm.PeriodicTask;

import java.util.concurrent.TimeUnit;

public class FlybitsContextPlugin implements ContextPlugin{

    public static final String EXTRA_NOTIFICATION = "flybits_extra_notification";
    public static final String EXTRA_MINIMUM_REFRESH_TIME = "flybits_minimum_refresh_time";

    private String _TAG = "FlybitsContextPlugin";

    private long timeInSecondsFlex;
    private long timeInSeconds;
    private Notification foregroundServiceNotification;
    private Bundle extras;
    private Class<? extends FlybitsContextPluginService> taskServiceBackground;

    /**
     * Constructor used to initialize {@code FlybitsContextPlugin}.
     *
     * @param builder The {@link FlybitsContextPlugin.Builder} where all the attributes about the
     *                {@code FlybitsContextPlugin} are initialized.
     */
    private FlybitsContextPlugin(FlybitsContextPlugin.Builder builder){
        if (builder.extras != null){
            extras = builder.extras;
        }

        if (builder.taskServiceBackground != null){
            taskServiceBackground = builder.taskServiceBackground;
        }

        timeInSecondsFlex = builder.timeInSecondsFlex;
        timeInSeconds = builder.timeInSeconds;
        foregroundServiceNotification = builder.foregroundServiceNotification;

        if (timeInSeconds != Builder.GRETZKYS_UNIQUE_ID && timeInSeconds < 60 && foregroundServiceNotification == null){
            throw new IllegalStateException("foregroundServiceNotification cannot be null if refresh time is less than 60 seconds. Use Builder.setForegroundServiceNotification().");
        }
    }

    @Override
    public boolean equals(Object o) {

        if (!(o instanceof FlybitsContextPlugin)){
            return false;
        }

        FlybitsContextPlugin another   = (FlybitsContextPlugin) o;
        return another.getService().getSimpleName().equals(getService().getSimpleName());
    }

    /**
     * Get the additional attributes associated to this {@code FlybitsContextPlugin}.
     *
     * @return The additional attributes bundled within a {@link Bundle} object that contains all
     * the additional attributes.
     */
    Bundle getExtras(){

        if (extras == null){
            return Bundle.EMPTY;
        }
        return extras;
    }

    /**
     * Get the refresh timer when this {@code FlybitsContextPlugin} should refresh its Context Data.
     *
     * @return The refresh rate for retrieving the Context Data of this {@code FlybitsContextPlugin}
     * in seconds.
     */
    long getRefreshTime(){
        return timeInSeconds;
    }

    /**
     * Get the allowed flex time that indicates the maximum that a {@code FlybitsContextPlugin} can
     * exceed its {@link #getRefreshTime()} before forcing itself to refresh.
     *
     * @return The flexible refresh time for retrieving the Context Data of this
     * {@code FlybitsContextPlugin} in seconds.
     */
    long getRefreshTimeFlex(){
        return timeInSecondsFlex;
    }

    /**
     * Determines if the application developer has indicated that the refresh time is less than 60
     * seconds if this is the case, then a custom service should be run, otherwise a
     * {@code GcmTaskService} will be used.
     *
     * @return true if a custom service should run based on a custom timer, false will indicate that
     * the {@code GcmTaskService} will be used.
     */
    boolean isCustomRefreshSet(){
        return timeInSeconds < 60;
    }

    /**
     * Get the {@link FlybitsContextPluginService} that is associated to this
     * {@link FlybitsContextPlugin} and should be periodically executed.
     *
     * @return The {@link FlybitsContextPluginService} that should be periodically executed.
     */
    public Class<? extends FlybitsContextPluginService> getService() {
        return taskServiceBackground;
    }

    @Override
    public void onStart(Context mContext) {
        Logger.setTag("FlybitsContextPlugin").d("onStart()");
        if (isCustomRefreshSet()){
            if (!isServiceRunning(mContext, getService())) {
                Intent intent = new Intent(mContext, getService());
                Bundle extras = setBundle();
                extras.putParcelable(EXTRA_NOTIFICATION, foregroundServiceNotification);
                intent.putExtras(extras);
                ContextCompat.startForegroundService(mContext, intent);
            }
        }else{

            SharedPreferences.Editor preferences   = ContextScope.getContextPreferences(mContext).edit();
            preferences.putLong(getPrefName(getService()), getRefreshTime());
            preferences.apply();

            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
            PeriodicTask.Builder task = new PeriodicTask.Builder()
                    .setTag(getService().getSimpleName())
                    .setExtras(setBundle())
                    .setPeriod(getRefreshTime())
                    .setUpdateCurrent(true)
                    .setPersisted(true)
                    .setFlex(getRefreshTimeFlex())
                    .setRequiredNetwork(PeriodicTask.NETWORK_STATE_ANY)
                    .setService(getService());
            mGcmNetworkManager.schedule(task.build());
        }

        Logger.setTag(_TAG).d("Activated: " + getService().getSimpleName());
    }

    @Override
    public void onStop(Context mContext) {
        Logger.setTag("FlybitsContextPlugin").d("onStop()");
        if (isCustomRefreshSet()) {

            if (isServiceRunning(mContext, getService())) {
                Intent intent = new Intent(mContext, getService());
                mContext.stopService(intent);
            }
        }else{
            Intent intent = new Intent(mContext, getService());
            if (ContextUtilities.isServiceDefined(mContext, intent)) {
                GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);
                try {
                    mGcmNetworkManager.cancelTask(getService().getSimpleName(), getService());
                } catch (IllegalArgumentException ex) {
                    Logger.exception("FlybitsContextPlugin.stop", ex);
                }
            }
        }
        Logger.setTag(_TAG).d("Deactivated: " + getService().getSimpleName());
    }

    @Override
    public void onRefresh(Context mContext) {
        Logger.setTag("FlybitsContextPlugin").d("onRefresh()");
        if (isCustomRefreshSet()) {
            if (!isServiceRunning(mContext, getService())) {
                Intent intent = new Intent(mContext, getService());
                Bundle extras = setBundle();
                extras.putParcelable(EXTRA_NOTIFICATION, foregroundServiceNotification);
                intent.putExtras(extras);
                ContextCompat.startForegroundService(mContext, intent);
            }
        }else{
            GcmNetworkManager mGcmNetworkManager = GcmNetworkManager.getInstance(mContext);

            OneoffTask.Builder task = new OneoffTask.Builder()
                    .setTag(getService().getSimpleName())
                    .setExtras(setBundle())
                    .setExecutionWindow(getRefreshTime(), getRefreshTimeFlex())
                    .setService(getService())
                    .setUpdateCurrent(true);
            mGcmNetworkManager.schedule(task.build());
        }
    }

    // Custom method to determine whether a service is running
    private boolean isServiceRunning(Context mContext, Class<?> serviceClass){
        ActivityManager activityManager = (ActivityManager) mContext.getSystemService(Context.ACTIVITY_SERVICE);

        // Loop through the running services
        for(ActivityManager.RunningServiceInfo service : activityManager.getRunningServices(Integer.MAX_VALUE)) {
            if (serviceClass.getName().equals(service.service.getClassName())) {
                // If the service is running then return true

                Logger.setTag(_TAG).d("unregisterUploadingContext Completed");
                Logger.setTag(_TAG).d("isServiceRunning: " + service.service.getClassName() + ", true");
                return true;
            }
        }

        Logger.setTag(_TAG).d("isServiceRunning: " + serviceClass.getName() + ", false");
        return false;
    }

    private Bundle setBundle(){
        Bundle bundle = (getExtras() == Bundle.EMPTY) ? new Bundle() : getExtras();
        bundle.putLong(EXTRA_MINIMUM_REFRESH_TIME, getRefreshTime());
        return bundle;
    }

    public static String getPrefName(Class classToBeSaved){
        return FlybitsContextPluginService.PREF_CP_REFRESH_RATE_START + classToBeSaved.getSimpleName();
    }

    public static class Builder{

        private static int GRETZKYS_UNIQUE_ID = -99;

        long timeInSeconds = 60;
        Notification foregroundServiceNotification = null;
        long timeInSecondsFlex = 60;
        Bundle extras;
        Class<? extends FlybitsContextPluginService> taskServiceBackground;

        public Builder(Class<? extends FlybitsContextPluginService> classObj){
            setService(classObj);
        }

        public Builder(ReservedContextPlugin plugin){
            switch (plugin){
                case BATTERY:
                    setService(BatteryContextPluginService.class);
                    break;
                case CARRIER:
                    setService(CarrierContextPluginService.class);
                    break;
                case LOCATION:
                    setService(LocationContextPluginService.class);
                    break;
                case WEATHER:
                    setService(WeatherContextPluginService.class);
                    break;
                case NETWORK_CONNECTIVITY:
                    setService(NetworkContextPluginService.class);
                    break;
                case ACTIVITY:
                    setService(ActivityContextPluginService.class);
                    break;
                case LANGUAGE:
                    setService(LanguageContextPluginService.class);
                    break;
                case FITNESS:   
                    setService(FitnessContextPluginService.class);
                    break;
                case BEACON:
                    setService(BeaconScanningService.class);
                    timeInSeconds   = GRETZKYS_UNIQUE_ID;
                    break;
            }
        }

        /**
         * Used to create the {@link FlybitsContextPlugin} based on the attributes set within the
         * {@code Builder}.
         *
         * @return The {@link FlybitsContextPlugin} object that can registered through the
         * {@link com.flybits.context.ContextManager#start(Context, ContextPlugin)}.
         *
         * @throws InvalidContextPluginException If the {@link #setService(Class)} is not set.
         */
        public FlybitsContextPlugin build() throws InvalidContextPluginException {
            if (timeInSeconds == 0){
                timeInSeconds = 60;
            }
            if (timeInSecondsFlex == 0){
                timeInSecondsFlex = 60;
            }
            return new FlybitsContextPlugin(this);
        }

        /**
         * Set additional bundle values that can be processed by the Background/Foreground services.
         * An example of this use case is the Location Context Plugin which allows the application
         * to define the minimum displacement for location updates. This displacement can be added
         * to the {@code extras} Bundle.
         *
         * @param extras The additional Bundle values that should be passed to the
         *               Background/Foreground service.
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setExtras(Bundle extras){
            this.extras = extras;
            return this;
        }

        /**
         * The refresh time in seconds which this {@link FlybitsContextPlugin} should wait before
         * refreshing its values.
         *
         * If this time is less than 60 seconds then the service will be ran in the foreground.
         * If this is the case the Notification to represent this service must be set using the
         * Builder.setNotification method.
         *
         * @param timeUnit The refresh rate in seconds.
         * @param flexTimeUnit The flexible refresh rate in seconds that this
         *                     {@link FlybitsContextPlugin} will refresh its data.
         * @param unit The unit of measure that time should be calculated in.
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setRefreshTime(long timeUnit, long flexTimeUnit, TimeUnit unit){

            if (timeInSeconds != GRETZKYS_UNIQUE_ID) {
                this.timeInSeconds = unit.toSeconds(timeUnit);
                this.timeInSecondsFlex = unit.toSeconds(flexTimeUnit);
            }
            return this;
        }

        /**
         * Set the Notification to be displayed while the foreground service associated with this
         * context plugin is running. Failure to provide notification when the refresh time is less
         * than 60 seconds will result in IllegalStateException. Will be unused if refresh time >= 60.
         *
         * @param foregroundServiceNotification Notification that will be displayed for this
         *                                      context plugin's foreground service.
         *
         * @return The {@code Builder} class which can be reused to add additional attributes to.
         */
        public Builder setForegroundServiceNotification(Notification foregroundServiceNotification) {
            this.foregroundServiceNotification = foregroundServiceNotification;
            return this;
        }

        void setService(Class<? extends FlybitsContextPluginService> classObj) {
            this.taskServiceBackground  = classObj;
        }
    }
}
