package com.flybits.context.plugins.location;

import android.Manifest;
import android.content.pm.PackageManager;
import android.location.Location;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.support.v4.content.ContextCompat;

import com.flybits.commons.library.logging.Logger;
import com.flybits.context.models.ContextData;
import com.flybits.context.services.FlybitsContextPluginService;
import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.api.GoogleApiClient;
import com.google.android.gms.location.LocationListener;
import com.google.android.gms.location.LocationRequest;
import com.google.android.gms.location.LocationServices;

import java.util.concurrent.TimeUnit;

public class LocationContextPluginService extends FlybitsContextPluginService implements
        GoogleApiClient.ConnectionCallbacks,
        GoogleApiClient.OnConnectionFailedListener,
        LocationListener {

    private LocationRequest mLocationRequest;
    private GoogleApiClient mGoogleApiClient;
    private final static String _TAG = "PluginLocation";
    private boolean isDestroyed;

    private long minDisplacement = -1;

    @Override
    public void initialize(Bundle bundle) {

        isDestroyed = false;
        if (mGoogleApiClient == null) {
            mGoogleApiClient = new GoogleApiClient.Builder(this)
                    .addApi(LocationServices.API)
                    .addConnectionCallbacks(this)
                    .addOnConnectionFailedListener(this)
                    .build();

            if (bundle != null && bundle.containsKey("minDisplacement")) {
                minDisplacement = bundle.getLong("minDisplacement");
            }
        }
    }

    @Override
    public boolean isSupported() {
        return Build.VERSION.SDK_INT < 23 ||
                ContextCompat.checkSelfPermission(getBaseContext(), Manifest.permission.ACCESS_FINE_LOCATION) == PackageManager.PERMISSION_GRANTED;
    }

    @Override
    public ContextData getData() {
        Logger.setTag(_TAG).d("Fetching Data ...");

        if (mGoogleApiClient != null) {
            if (!mGoogleApiClient.isConnected()) {
                mGoogleApiClient.blockingConnect(10, TimeUnit.SECONDS);
            }

            if (!mGoogleApiClient.isConnected()) {
                wait(10, TimeUnit.SECONDS);
            }

            if (mGoogleApiClient.isConnected()) {
                Location mLastLocation = LocationServices.FusedLocationApi.getLastLocation(mGoogleApiClient);
                if (mLastLocation != null) {

                    Logger.setTag(_TAG).d("Last Known Location: " + mLastLocation);
                    return new LocationData(mLastLocation);
                }
            }
        }
        return null;
    }

    @Override
    public void onConnected(@Nullable Bundle bundle) {

        if (isSupported() && !isDestroyed) {
            mLocationRequest = LocationRequest.create()
                    .setPriority(LocationRequest.PRIORITY_HIGH_ACCURACY)
                    .setFastestInterval(getRefreshTime() * 1000)
                    .setInterval(getRefreshTime() * 1000);

            Logger.setTag(_TAG).d("Google API Services connected!");
            LocationServices.FusedLocationApi.requestLocationUpdates(mGoogleApiClient, mLocationRequest, this);
        }else{
            stopSelf();
        }
    }

    @Override
    public void onConnectionSuspended(int i) {
        Logger.setTag(_TAG).w("GoogleAPIClient: onConnectionSuspended: " + i);
        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                mGoogleApiClient.connect();
            }
        }, 1000);
    }

    @Override
    public void onConnectionFailed(@NonNull ConnectionResult connectionResult) {
        Logger.setTag(_TAG).e("GoogleAPIClient: onConnectionFailed: + " + connectionResult);
    }

    @Override
    public void onLocationChanged(Location location) {
        Logger.setTag(_TAG).d("On Location Changed: " + location + ", " + location.getProvider());
        LocationData data       = new LocationData(location);
        long timeInSec              = (data.getTime() > 0) ? data.getTime() : (System.currentTimeMillis() / 1000);
        data.update(getBaseContext(), timeInSec, null);
    }

    @Override
    public void onDestroy() {
        Logger.setTag(_TAG).d("...Destroyed");
        isDestroyed = true;

        if (mGoogleApiClient != null && mGoogleApiClient.isConnected()) {
            LocationServices.FusedLocationApi.removeLocationUpdates(mGoogleApiClient, this);
            mGoogleApiClient.disconnect();
        }

        super.onDestroy();
    }

    @Override
    public String[] getRequiredPermissions() {
        return new String[]{"android.permission.ACCESS_FINE_LOCATION", "android.permission.ACCESS_COARSE_LOCATION"};
    }

    public GoogleApiClient getGoogleApiClient()
    {
        return mGoogleApiClient;
    }
}
