package com.flybits.context.models.internal;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.content.Context;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.flybits.commons.library.api.FlyAway;
import com.flybits.commons.library.api.results.callbacks.PagedResultCallback;
import com.flybits.commons.library.exceptions.FlybitsException;
import com.flybits.commons.library.models.internal.PagedResponse;
import com.flybits.commons.library.models.internal.QueryParameters;
import com.flybits.commons.library.models.internal.Result;
import com.flybits.context.ContextScope;
import com.flybits.context.deserializations.DeserializeContextPlugins;
import com.flybits.context.models.results.ContextPluginResult;

import java.util.HashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * The {@code Plugin} class is responsible for retrieving all {@code ContextPlugins} that can be
 * enabled for this {@code Project}. This includes both pre-set public {@code ContextPlugins} such
 * as {@code Battery}, {@code Activity}, {@code Location}, etc. It also includes custom
 * {@code ContextPlugins} that have been created by the Administrator.
 */
@Entity(tableName = "plugin")
public class Plugin {

    public static final String API = ContextScope.ROOT+"/ctxschema/contextplugins";

    @PrimaryKey
    @ColumnInfo(name = "contextID")
    @NonNull
    private String id;

    @ColumnInfo(name = "refreshTime")
    private long refreshRate;

    @ColumnInfo(name = "isSupported")
    private boolean isSupported;

    @ColumnInfo(name = "className")
    private String className;

    @ColumnInfo(name = "isEnabled")
    private boolean isEnabled;

    @Ignore
    private String name;

    @Ignore
    private boolean isReserved;

    @Ignore
    private HashMap<String, String> listOfAttributes;

    public Plugin(){}

    /**
     * Default constructor that defines all the necessary attributes associated to a valid
     * {@code Plugin}.
     *
     * @param id The unique identifier associated to the {@code Plugin}.
     * @param name The name of the {@code Plugin}.
     * @param refreshRate Indicates how often, in seconds, the {@code Plugin} should be refreshed.
     * @param isEnabled Indicates whether or not the {@code Plugin} is enabled.
     * @param isReserved Indicates whether or not the {@code Plugin} is a Plugin that is created
     *                   by Flybits as Public {@code Plugin}.
     * @param isSupported Indicates that this {@code Plugin} is supported by Android devices.
     */
    public Plugin(@NonNull String id, @NonNull String name, long refreshRate, boolean isEnabled,
                  boolean isReserved, boolean isSupported) {
        this(id, name, refreshRate, isEnabled, isReserved, isSupported, new HashMap<String, String>());
    }

    /**
     * Default constructor that defines all the necessary attributes associated to a valid
     * {@code Plugin}.
     *
     * @param id The unique identifier associated to the {@code Plugin}.
     * @param name The name of the {@code Plugin}.
     * @param refreshRate Indicates how often, in seconds, the {@code Plugin} should be refreshed.
     * @param isEnabled Indicates whether or not the {@code Plugin} is enabled.
     * @param isReserved Indicates whether or not the {@code Plugin} is a Plugin that is created
     *                   by Flybits as Public {@code Plugin}.
     * @param isSupported Indicates that this {@code Plugin} is supported by Android devices.
     * @param listOfAttributes List of attributes associated to the Context Plugin.
     */
    public Plugin(@NonNull String id, @NonNull String name, long refreshRate, boolean isEnabled,
                  boolean isReserved, boolean isSupported, HashMap<String, String> listOfAttributes) {
        this.id                 = id;
        this.isEnabled          = isEnabled;
        this.isReserved         = isReserved;
        this.name               = name;
        this.refreshRate        = refreshRate;
        this.isSupported        = isSupported;
        this.listOfAttributes   = listOfAttributes;
    }

    /**
     * A request to get a list of {@code Rule}s based on the {@code QueryParameters} parameter which
     * defines which options should be used to filter the list of returned parameters.
     *
     * @param mContext The context of the application.
     * @param params The {@code QueryParameters} that indicate filters that should be placed on
     *               the request.
     * @param callback The callback that indicates whether or not the request was successful or
     *                 whether there was an error.
     *
     * @return The {@code RulesResult} that contains the network request for getting a list of
     * {@code Rule}s.
     */
    public static ContextPluginResult get(@NonNull final Context mContext, @NonNull final QueryParameters params,
                                          @NonNull PagedResultCallback<Plugin> callback){

        final Handler handler = new Handler(Looper.getMainLooper());
        final ExecutorService executorService = Executors.newSingleThreadExecutor();
        final ContextPluginResult result = new ContextPluginResult(mContext, params, callback, executorService, handler);
        executorService.execute(new Runnable() {
            public void run() {
                try{

                    DeserializeContextPlugins deserializer     = new DeserializeContextPlugins();
                    final Result<PagedResponse<Plugin>> getPlugins = FlyAway.get(mContext, API, params, deserializer, "Plugin.get");
                    result.setResult(getPlugins);
                }catch (final FlybitsException e){
                    result.setResult(new Result<PagedResponse<Plugin>>(e,"Plugin.get() failed"));
                }
            }
        });
        return result;
    }

    /**
     * Return the map of Context Plugins attributes associated to the Context Plugin
     *
     * @return A hashmap consisting of the context id as key and value type as the value.
     */
    public @Nullable HashMap<String, String> getAttributes(){
        return listOfAttributes;
    }

    /**
     * Get the package path including the name of the class that should be run as part of retrieving
     * information associated to the Context Plugin.
     *
     * @return The String representation of the class name with the package path as a prefix.
     */
    public @Nullable String getClassName() {
        return className;
    }

    /**
     * Get the unique identifier for this {@code ContextPlugin}.
     *
     * @return The String representation of the unique identifier.
     */
    public String getId() {
        return id;
    }

    /**
     * Get the name of the {@code Plugin}.
     *
     * @return The String representing the name of the {@code Plugin}.
     */
    public String getName() {
        return name;
    }

    /**
     * Get the rate at which the {@code Plugin} should refresh it's data.
     *
     * @return The time in seconds that indicates how often {@code Plugin} should refresh it's data,
     * -1 can be returned which indicates that data should never be refreshed.
     */
    public long getRefreshRate() {
        return refreshRate;
    }

    /**
     * Indicates whether or not the {@code Plugin} is supported on Android devices. The
     * {@code Plugin} must be enabled and supported on Android devices.
     *
     * @return true if the {@code Plugin} is supported on Android devices, false otherwise.
     */
    public boolean isSupported() {
        return isEnabled && isSupported;
    }

    /**
     * Indicates whether or not the {@code Plugin} should enabled on Android devices.
     *
     * @return true if the {@code Plugin} is enabled on Android devices, false otherwise.
     */
    public boolean isEnabled() {
        return isEnabled;
    }

    /**
     * Indicates whether or not the {@code Plugin} is reserved, meaning is it a Public
     * {@code Plugin} created by Flybits and available to all {@code Projects}, or was it custom
     * {@code Plugin} created by the administrator of the {@code Project}.
     *
     * @return true if the {@code Plugin} has been reserved (created by Flybits as a public
     * {@code Plugin}), false otherwise.
     */
    public boolean isReserved() {
        return isReserved;
    }

    /**
     * Sets the name of the Context Plugin class that should be automatically run by SDK in order to
     * collect the information about the Context Plugin.
     *
     * @param className The string representation of the class name including the package(path).
     */
    public void setClassName(String className){
        this.className  = className;
    }

    /**
     * Set whether or not the {@code Plugin} is enabled through the Developer Portal.
     *
     * @param isEnabled true if the {@code Plugin} is enabled, false otherwise.
     */
    public void setEnabled(boolean isEnabled) {
        this.isEnabled = isEnabled;
    }

    /**
     * Set the {@code Plugin} identifier for this Context Plugin.
     *
     * @param id The string representation of the Context plugin unique identifier.
     */
    public void setId(String id) {
        this.id = id;
    }

    /**
     * Set how often the Context Plugin should refresh it's data through the SDK. The mechanism that
     * is used is to have a timeout based on the {@code refreshRate}.
     *
     * @param refreshRate The time in seconds to refresh the Context Plugin's data.
     */
    public void setRefreshRate(long refreshRate) {
        this.refreshRate = refreshRate;
    }

    /**
     * Set whether or not the {@code Plugin} is supported through by the Android device
     *
     * @param supported true if the Android SDK supports the Context Plugin, false otherwise.
     */
    public void setSupported(boolean supported) {
        isSupported = supported;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Plugin plugin = (Plugin) o;

        return id != null ? id.equals(plugin.id) : plugin.id == null;
    }
}
