package com.flybits.context.plugins.beacon;

import android.arch.persistence.room.ColumnInfo;
import android.arch.persistence.room.Entity;
import android.arch.persistence.room.Ignore;
import android.arch.persistence.room.PrimaryKey;
import android.os.Parcel;
import android.os.Parcelable;
import android.support.annotation.NonNull;

import com.flybits.context.models.ContextData;

import org.altbeacon.beacon.Beacon;
import org.json.JSONException;
import org.json.JSONObject;

import static com.flybits.context.plugins.beacon.BeaconType.Eddystone;
import static com.flybits.context.plugins.beacon.BeaconType.IBeacon;

/**
 * The {@code BeaconActive} class is responsible for providing information about beacons that are in
 * range of the device. Information about the beacon including their identifiers, and the type of
 * beacon it is.
 */
@Entity(tableName = "beaconsActive")
public class BeaconActive extends ContextData implements Parcelable {

    @ColumnInfo(name = "id1")
    private String id1;

    @ColumnInfo(name = "id2")
    private String id2;

    @ColumnInfo(name = "id3")
    private String id3;

    @ColumnInfo(name = "type")
    private BeaconType type;

    @ColumnInfo(name = "lastSeen")
    private long lastSeen;

    @ColumnInfo(name = "isSent")
    private boolean isSent;

    @ColumnInfo(name = "isInRange")
    private boolean isInRange;

    @ColumnInfo(name = "id")
    @PrimaryKey
    @NonNull
    private String compositeID;

    /**
     * Default constructor needed for generics instantiation
     */
    public BeaconActive(){}

    /**
     * Constructor used for un-flattening a {@code BeaconActive} parcel.
     *
     * @param in the parcel that contains the un-flattened {@code BeaconActive} parcel.
     */
    @Ignore
    protected BeaconActive(Parcel in){
        id1     = in.readString();
        id2     = in.readString();
        type    = BeaconType.fromValue(in.readString());
        switch (type){
            case IBeacon:
                id3     = in.readString();
                break;
        }
        isSent      = in.readInt() == 1;
        isInRange   = in.readInt() == 1;
        lastSeen    = in.readLong();

        String compositeID = id1;
        if (id2 != null){
            compositeID += ("_" + id2);
        }

        if (type == IBeacon && id3 != null){
            compositeID += ("_" + id3);
        }
        this.compositeID =compositeID;
    }

    /**
     * Constructor used define a {@code BeaconActive} object based on preset values.
     *
     * @param type Indicates which type of beacon this is; EddyStone or iBeacon.
     * @param id1 The UUID (iBeacon) or namespace (EddyStone)
     * @param id2 The majorID (iBeacon) or instance (EddyStone)
     * @param id3 The minorID (iBeacon). For EddyStone this value will be null.
     * @param lastSeen The epoch time of when this beacon was detected.
     * @throws IllegalStateException is thrown whenever invalid beacon ids are entered. For iBeacon,
     * {@code id1}, {@code id2}, and {@code id3} cannot be null. For EddyStone, {@code id1} and
     * {@code id2} cannot be null.
     */
    @Ignore
    public BeaconActive(@NonNull BeaconType type, @NonNull String id1, String id2, String id3, long lastSeen)
            throws IllegalStateException{

        setBeaconType(type, id1, id2, id3);
        this.lastSeen   = lastSeen;
        isSent          = false;

        String compositeID = id1;
        if (id2 != null){
            compositeID += ("_" + id2);
        }

        if (type == IBeacon && id3 != null){
            compositeID += ("_" + id3);
        }
        this.compositeID =compositeID;
    }

    /**
     * Constructor used define a {@code BeaconActive} object based on a {@code Beacon} object.
     *
     * @param beacon The {@code Beacon} object used to define the {@code BeaconActive}.
     */
    @Ignore
    public BeaconActive(Beacon beacon){

        id1 = beacon.getId1().toString();
        if (beacon.getIdentifiers().size() > 1){
            id2     = beacon.getId2().toString();
        }

        if (beacon.getIdentifiers().size() > 2){
            id3     = beacon.getId3().toString();
            type    = BeaconType.IBeacon;
        }else{
            type    = BeaconType.Eddystone;
        }
        this.lastSeen   = System.currentTimeMillis() / 1000;
        isSent          = false;


        String compositeID = id1;
        if (id2 != null){
            compositeID += ("_" + id2);
        }

        if (type == IBeacon && id3 != null){
            compositeID += ("_" + id3);
        }
        this.compositeID =compositeID;
    }

    /**
     * Get the a single unique identifier for the Beacon which is comprised a combination of
     * UUID/MajorID/MinorID for iBeacon and Namespace/Instance for EddyStone separated by the "_"
     * delimiter.
     *
     * @return The String representation as either iBeacon (UUID_MajorID_MinorID) or EddyStone
     * (Namespace_Instance).
     */
    public String getCompositeID() {
        return compositeID;
    }

    /**
     * Get the UUID or Namespace identifier depending on which {@link #getType()} the beacon is.
     *
     * @return The UUID for an iBeacon type or Namespace for an EddyStone type beacon.
     */
    public String getId1() {
        return id1;
    }

    /**
     * Get the MajorID or Instance identifier depending on which {@link #getType()} the beacon is.
     *
     * @return The MajorID for an iBeacon type or Instance for an EddyStone type beacon.
     */
    public String getId2() {
        return id2;
    }

    /**
     * Get the MinorID for an iBeacon type. For EddyStone beacons this should be null.
     *
     * @return The MinorID for an iBeacon type or null for EddyStone.
     */
    public String getId3() {
        return id3;
    }

    /**
     * Get the epoch time when this {@code BeaconActive} was last detected.
     *
     * @return The epoch time when this {@code BeaconActive} was last detected.
     */
    public long getLastSeen() {
        return lastSeen;
    }

    /**
     * Get the {@link com.flybits.context.plugins.beacon.BeaconType} that this beacon is. Currently,
     * the options include iBeacon and EddyStone.
     *
     * @return The {@link com.flybits.context.plugins.beacon.BeaconType} of this beacon.
     */
    public BeaconType getType() {
        return type;
    }

    /**
     * Indicates whether or not this Beacon is within range of the device.
     *
     * @return true if the Beacon is within range of the device, false otherwise.
     */
    public boolean isInRange(){
        return isInRange;
    }

    /**
     * Indicates whether or not the Beacon information has been sent to the server.
     *
     * @return true if the Beacon information is sent to the server, false otherwise.
     */
    public boolean isSent() {return isSent;}

    /**
     * Set the composite identifier of the {@code BeaconActive}. For {@link BeaconType#Eddystone}
     * will be in the following structure: id1-id2 and for {@link BeaconType#IBeacon} will be in the
     * following structure: id1-id2-id3.
     *
     * @param compositeID The composite identifier of the {@code BeaconActive}.
     */
    public void setCompositeID(String compositeID) {
        this.compositeID = compositeID;
    }

    /**
     * Sets the majorID of the {@link BeaconType#IBeacon} or the {@code namespace} of the
     * {@link BeaconType#Eddystone}.
     *
     * @param id1 The majorID of the {@link BeaconType#IBeacon} or the {@code namespace} of the
     * {@link BeaconType#Eddystone}.
     */
    public void setId1(String id1) {
        this.id1 = id1;
    }

    /**
     * Sets the minorID of the {@link BeaconType#IBeacon} or the {@code instanceID} of the
     * {@link BeaconType#Eddystone}.
     *
     * @param id2 The minorID of the {@link BeaconType#IBeacon} or the namespace of the
     * {@link BeaconType#Eddystone}.
     */
    public void setId2(String id2) {
        this.id2 = id2;
    }

    /**
     * Sets the unique identifier of the {@link BeaconType#IBeacon} or null of the
     * {@link BeaconType#Eddystone}.
     *
     * @param id3 The unique identifier of the {@link BeaconType#IBeacon} or null of the
     * {@link BeaconType#Eddystone}.
     */
    public void setId3(String id3) {
        this.id3 = id3;
    }

    /**
     * Sets whether or not the Beacon is in range of the device.
     *
     * @param isRange The boolean value which indicates whether or not Beacon is within range of the
     *                device.
     */
    public void setInRange(boolean isRange){
        this.isInRange  = isRange;
    }

    /**
     * Sets whether or not the Beacon is sent to the server for processing.
     *
     * @param isSent The boolean value which indicates whether or not the Beacon information was
     *               sent to the server.
     */
    public void setIsSent(boolean isSent){
        this.isSent  = isSent;
    }

    /**
     * Indicates when the {@code BeaconActive} was last seen by the Beacon monitoring service.
     *
     * @param lastSeen The epoch time when the {@code BeaconActive} was last seen.
     */
    public void setLastSeen(long lastSeen) {
        this.lastSeen = lastSeen;
    }

    /**
     * Sets the epoch time of when the Beacon was last seen.
     *
     * @param timestamp The epoch time value of when the Beacon was last seen.
     */
    public void setTimestamp(long timestamp){
        this.lastSeen  = timestamp;
    }

    /**
     * Set the {@link BeaconType} that indicates which type of {@code Beacon} this
     * {@code BeaconActive} is for.
     *
     * @param type The {@link BeaconType} for the {@code BeaconActive}.
     */
    public void setType(BeaconType type) {
        this.type = type;
    }

    @Override
    public String getPluginID() {
        return "ctx.sdk.beacon";
    }

    @Override
    public String toString() {
        return toJson();
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {

        dest.writeString(id1);
        dest.writeString(id2);
        dest.writeString(type.getValue());
        switch (type){
            case IBeacon:
                dest.writeString(id3);
                break;
        }
        dest.writeInt(isSent? 1 : 0);
        dest.writeInt(isInRange? 1 : 0);
        dest.writeLong(lastSeen);
    }

    /**
     * Parcelable.Creator that instantiates {@code BeaconActive} objects
     */
    public static final Creator<BeaconActive> CREATOR = new Creator<BeaconActive>() {
        @Override
        public BeaconActive createFromParcel(Parcel in) {
            return new BeaconActive(in);
        }

        @Override
        public BeaconActive[] newArray(int size) {
            return new BeaconActive[size];
        }
    };

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        BeaconActive that = (BeaconActive) o;
        return getCompositeID().equalsIgnoreCase(that.getCompositeID());
    }

    @Override
    public void fromJson(String json) {

        try {
            JSONObject jsonObj  = new JSONObject(json);
            type                = BeaconType.fromValue(jsonObj.getString("type"));
            isInRange           = jsonObj.getBoolean("inRange");
            if (!jsonObj.isNull("majorID")){
                id2 = jsonObj.getString("majorID");
                id3 = jsonObj.getString("minorID");
                id1 = jsonObj.getString("uuid");
            }else{
                id2 = jsonObj.getString("instance");
                id1 = jsonObj.getString("namespace");
            }
        }catch (JSONException exception){}

    }

    @Override
    public String toJson() {
        JSONObject object = new JSONObject();
        try {

            switch (type)
            {
                case IBeacon:
                    object.put("majorID", id2);
                    object.put("minorID", id3);
                    object.put("uuid", id1);
                    break;
                case Eddystone:
                    object.put("instance", id2);
                    object.put("namespace", id1);
                    break;
            }

            object.put("type", type.getValue());
            object.put("inRange", isInRange());
        }catch (JSONException exception){}

        return object.toString();
    }

    private void setBeaconType(BeaconType type, String id1, String id2, String id3)
        throws IllegalStateException{

        if (type == Eddystone && (id1 == null || id2 == null)){
            throw new IllegalStateException("You have entered an invalid EddyStone Beacon");
        }

        if (type == IBeacon && (id1 == null || id2 == null || id3 == null)){
            throw new IllegalStateException("You have entered an invalid iBeacon Beacon");
        }

        switch (type){
            case Eddystone:
                this.id1    = id1;
                this.id2    = id2;
                break;
            case IBeacon:
                this.id1    = id1;
                this.id2    = id2;
                this.id3    = id3;
                break;
        }
        this.type       = type;
    }
}
