package com.flybits.context.models;

/**
 * Indicates the scope that a {@link Rule} is associated to. Currently Rules can have either
 * {@link #USER} or {@link #PROJECT} scope. {@link #USER} scope is used to define {@link Rule}s that
 * are only evaluated for the current user, while {@link #PROJECT} scope {@link Rule}s are evaluated
 * for all user's within the project.
 */
public enum RuleScope {
    /**
     * Indicates the "tenant" scope for a {@link Rule}.
     */
    PROJECT("tenant"),

    /**
     * Indicates the "user" scope for a {@link Rule}.
     */
    USER("user"),

    /**
     * Indicates that an unrecognized scope was set for the scope.
     */
    UNKNOWN("");

    private final String key;

    /**
     * Constructor that defines the key for each {@code RuleScope} option.
     *
     * @param key the string value representing each {@code RuleScope} option.
     */
    RuleScope(String key) {
        this.key = key;
    }

    /**
     * Get the string representation for the {@code RuleScope} option.
     *
     * @return string representation of the {@code RuleScope} option.
     */
    public String getKey() {
        return this.key;
    }

    /**
     * Get the {@code RuleScope} enum value corresponding to an String representation.
     *
     * @param key the string representation of the {@code RuleScope} enum.
     *
     * @return The {@code RuleScope} enum for the String representation.
     */
    public static RuleScope fromKey(String key) {
        for(RuleScope type : RuleScope.values()) {
            if(type.getKey().equalsIgnoreCase( key)) {
                return type;
            }
        }
        return UNKNOWN;
    }
}
